package ru.yandex.chemodan.sql;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import lombok.SneakyThrows;
import org.intellij.lang.annotations.Language;
import org.jetbrains.annotations.NotNull;
import org.springframework.jdbc.BadSqlGrammarException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.db.admin.sql.SqlAdminDataSource;
import ru.yandex.commune.db.admin.sql.SqlAdminManager;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.lang.StringUtils;


@ActionContainer
public class SqlWithHighlightAdminPage {

    @Language("SQL")
    public static final String GET_TABLE_COLUMN_SQL = "SELECT table_name, column_name, data_type FROM " +
            "information_schema.columns"
            + " WHERE table_schema not in ('information_schema', 'pg_catalog')"
            + " ORDER BY table_name, ordinal_position";
    private final SqlAdminManager sqlAdminManager;
    private final Cache<String, DbMetadataPojo> cacheMetadata;

    public SqlWithHighlightAdminPage(SqlAdminManager sqlAdminManager) {
        this.sqlAdminManager = sqlAdminManager;
        this.cacheMetadata = CacheBuilder.newBuilder()
                .expireAfterWrite(5, TimeUnit.MINUTES)
                .build();
    }

    @NotNull
    private DbMetadataPojo getMetadataPojo(SqlAdminDataSource dataSource) {
        Map<String, DbMetadataPojo.TableMetadataPojo> metadata = dataSource
                .find(GET_TABLE_COLUMN_SQL,
                        (rs, rowNum) -> new Tuple3<>(
                                rs.getString("table_name"),
                                rs.getString("column_name"),
                                rs.getString("data_type"))
                )
                .groupByMapValues(k -> k._1, v -> new Tuple2<>(v._2, v._3))
                .mapValues(listF -> {
                    ListF<DbMetadataPojo.ColumnMetadataPojo> columns =
                            listF.map(DbMetadataPojo.ColumnMetadataPojo.consF());
                    return new DbMetadataPojo.TableMetadataPojo(columns);
                });

        return new DbMetadataPojo(metadata);
    }

    @ZAction(defaultAction = true, timeout = 1, timeoutUnit = TimeUnit.MINUTES)
    @Path(value = "/sql_beta", methods = {HttpMethod.GET, HttpMethod.POST})
    public FindAllPojo findAll(
            @RequestParam(value = "query", required = false)
            String query,
            @RequestParam(value = "ds", required = false) final String ds) throws ExecutionException {

        ListF<String> dataSourceNames = sqlAdminManager.getDataSourceNames();
        DataSourcesPojo dataSourcesPojo = new DataSourcesPojo(dataSourceNames);
        Map<String, DbMetadataPojo> metadata = dataSourceNames.toMapMappingToValue(this::dataSourceMetadata);

        if (StringUtils.isBlank(ds) || StringUtils.isEmpty(query)) {
            return new FindAllPojo(dataSourcesPojo, metadata);
        }
        SqlAdminDataSource zds = sqlAdminManager.getDataSource(ds);
        try {

            ListF<List<Tuple2<String, Object>>> data = zds.find(query, new ColumnTableTupleRowMapper());
            return new FindAllPojo(dataSourcesPojo, query, data.map(RowPojo.consF()), metadata);
        } catch (BadSqlGrammarException e) {
            return new FindAllPojo(dataSourcesPojo, query, e.getCause().getMessage(), metadata);
        }
    }

    @ZAction(timeout = 1, timeoutUnit = TimeUnit.MINUTES)
    @Path(value = "/sql_beta/execute", methods = {HttpMethod.GET, HttpMethod.POST})
    public ExecutePojo execute(
            @RequestParam(value = "query", required = false)
            String query,
            @RequestParam(value = "ds", required = false) final String ds) throws ExecutionException {
        ListF<String> dataSourceNames = sqlAdminManager.getExecutableDataSourceNames();
        DataSourcesPojo dataSourcesPojo = new DataSourcesPojo(dataSourceNames);
        Map<String, DbMetadataPojo> metadata = dataSourceNames.toMapMappingToValue(this::executableDataSourceMetadata);

        if (StringUtils.isBlank(ds) || StringUtils.isEmpty(query)) {
            return new ExecutePojo(dataSourcesPojo, metadata);
        }
        SqlAdminDataSource zds = sqlAdminManager.getExecutableDataSource(ds);
        try {
            int result = zds.execute(query);
            return new ExecutePojo(dataSourcesPojo, query, result, metadata);
        } catch (BadSqlGrammarException e) {
            return new ExecutePojo(dataSourcesPojo, query, e.getCause().getMessage(), metadata);
        }
    }

    @SneakyThrows
    private DbMetadataPojo dataSourceMetadata(String s) {
        return cacheMetadata.get(s, () -> getMetadataPojo(sqlAdminManager.getDataSource(s)));
    }

    @SneakyThrows
    private DbMetadataPojo executableDataSourceMetadata(String s) {
        return cacheMetadata.get(s, () -> getMetadataPojo(sqlAdminManager.getExecutableDataSource(s)));
    }

    @Bendable
    private static final class ItemPojo {
        @BenderPart
        public final String column;
        @BenderPart
        public final String value;

        private ItemPojo(String column, String value) {
            this.column = column;
            this.value = value;
        }

        private static Function<Tuple2<String, Object>, ItemPojo> consF() {
            return (t) -> {
                String column = t._1;
                Object value = t._2;
                return new ItemPojo(column, value == null ? "NULL" : value.toString());
            };
        }
    }

    @Bendable
    private static final class RowPojo {
        @BenderPart
        public final ListF<ItemPojo> items;

        private RowPojo(ListF<ItemPojo> items) {
            this.items = items;
        }

        private static Function<List<Tuple2<String, Object>>, RowPojo> consF() {
            return map -> new RowPojo(Cf.x(map).map(ItemPojo.consF()));
        }
    }

    @XmlRootElement(name = "content")
    private static final class ExecutePojo {
        @BenderPart
        public final Option<String> query;
        @BenderPart
        public final Option<String> error;
        @BenderPart
        @BenderFlatten
        public final DataSourcesPojo dataSourcesPojo;
        @BenderPart
        public final Option<Integer> rowsAffected;

        @BenderPart
        public final Map<String, DbMetadataPojo> metadata;

        private ExecutePojo(DataSourcesPojo dataSourcesPojo, Map<String, DbMetadataPojo> metadata) {
            this.dataSourcesPojo = dataSourcesPojo;
            this.metadata = metadata;
            this.query = Option.empty();
            this.rowsAffected = Option.empty();
            this.error = Option.empty();
        }

        private ExecutePojo(DataSourcesPojo dataSourcesPojo, String query, int rowsAffected, Map<String,
                DbMetadataPojo> metadata) {
            this.dataSourcesPojo = dataSourcesPojo;
            this.query = Option.of(query);
            this.rowsAffected = Option.of(rowsAffected);
            this.metadata = metadata;
            this.error = Option.empty();
        }

        private ExecutePojo(DataSourcesPojo dataSourcesPojo, String query, String error,
                            Map<String, DbMetadataPojo> metadata) {
            this.dataSourcesPojo = dataSourcesPojo;
            this.query = Option.of(query);
            this.metadata = metadata;
            this.rowsAffected = Option.empty();
            this.error = Option.of(error);
        }
    }

    @XmlRootElement(name = "content")
    private static final class FindAllPojo {
        @BenderPart
        public final Option<String> query;
        @BenderPart
        public final Option<String> error;
        @BenderPart
        @BenderFlatten
        public final DataSourcesPojo dataSourcesPojo;
        @BenderPart(name = "row", wrapperName = "rows")
        public final ListF<RowPojo> data;

        @BenderPart
        public final Map<String, DbMetadataPojo> metadata;

        private FindAllPojo(DataSourcesPojo dataSourcesPojo, Map<String, DbMetadataPojo> metadata) {
            this.dataSourcesPojo = dataSourcesPojo;
            this.metadata = metadata;
            this.query = Option.empty();
            this.data = Cf.list();
            this.error = Option.empty();
        }

        private FindAllPojo(DataSourcesPojo dataSourcesPojo, String query, ListF<RowPojo> data, Map<String,
                DbMetadataPojo> metadata) {
            this.dataSourcesPojo = dataSourcesPojo;
            this.query = Option.of(query);
            this.data = data;
            this.metadata = metadata;
            this.error = Option.empty();
        }

        private FindAllPojo(DataSourcesPojo dataSourcesPojo, String query, String error,
                            Map<String, DbMetadataPojo> metadata) {
            this.dataSourcesPojo = dataSourcesPojo;
            this.query = Option.of(query);
            this.metadata = metadata;
            this.data = Cf.list();
            this.error = Option.of(error);
        }
    }


    @XmlRootElement(name = "content")
    private static final class DbMetadataPojo {
        @BenderPart(name = "table", wrapperName = "tables")
        public final Map<String, TableMetadataPojo> tables;

        private DbMetadataPojo(Map<String, TableMetadataPojo> tables) {
            this.tables = tables;
        }

        @XmlRootElement(name = "content")
        public static class TableMetadataPojo {

            @BenderPart(name = "column", wrapperName = "columns")
            public final ListF<ColumnMetadataPojo> columns;

            public TableMetadataPojo(ListF<ColumnMetadataPojo> columns) {
                this.columns = columns;
            }
        }

        @Bendable
        private static final class ColumnMetadataPojo {
            @BenderPart
            private final String name;
            @BenderPart
            public final String type;


            public String getName() {
                return name;
            }

            private ColumnMetadataPojo(String name, String type) {
                this.name = name;
                this.type = type;
            }

            private static Function<Tuple2<String, String>, ColumnMetadataPojo> consF() {
                return (t) -> new ColumnMetadataPojo(t._1, t._2);
            }
        }
    }


    @XmlRootElement(name = "content")
    private static final class DataSourcesPojo {
        @BenderPart(name = "data-source", wrapperName = "data-sources")
        public final ListF<String> dataSources;

        private DataSourcesPojo(ListF<String> dataSources) {
            this.dataSources = dataSources;
        }
    }
}
