package ru.yandex.chemodan.util;

import lombok.ToString;
import org.junit.Assert;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderTextValue;

@Bendable
@ToString
public class AppVersionRange {

    public static final String XIVA_PARAMETER_INTERVAL_DELIMITER = "-";

    private final Option<AppVersion> from;

    private final Option<AppVersion> to;

    public AppVersionRange(AppVersion from, AppVersion to) {
        Assert.assertTrue(from.compareTo(to) < 0);
        this.from = Option.of(from);
        this.to = Option.of(to);
    }

    protected AppVersionRange(Option<AppVersion> from, Option<AppVersion> to) {
        this.from = from;
        this.to = to;
    }

    public boolean isInRangeBothInclusive(AppVersion appVersion) {
        return greaterThanOrEqualFrom(appVersion) && lessThanOrEqualTo(appVersion);
    }

    protected boolean greaterThanOrEqualFrom(AppVersion appVersion) {
        return from.get().compareTo(appVersion) <= 0;
    }

    protected boolean lessThanOrEqualTo(AppVersion appVersion) {
        return to.get().compareTo(appVersion) >= 0;
    }

    public String asXivaParameterString() {
        return from.map(AppVersion::asXivaParameterString).getOrElse("") + XIVA_PARAMETER_INTERVAL_DELIMITER +
            to.map(AppVersion::asXivaParameterString).getOrElse("");
    }

    public static AppVersionRange fromIntervalString(String intervalString) {
        if (intervalString.endsWith(XIVA_PARAMETER_INTERVAL_DELIMITER)) {
            return new FromOnlyAppVersionRange(new AppVersion(intervalString
                .replaceAll(XIVA_PARAMETER_INTERVAL_DELIMITER, "")));
        }
        if (intervalString.startsWith(XIVA_PARAMETER_INTERVAL_DELIMITER)) {
            return new ToOnlyAppVersionRange(new AppVersion(intervalString
                .replaceAll(XIVA_PARAMETER_INTERVAL_DELIMITER, "")));
        }
        String[] versions = intervalString.split(XIVA_PARAMETER_INTERVAL_DELIMITER);
        Assert.assertTrue(versions.length == 2);
        return new AppVersionRange(new AppVersion(versions[0]), new AppVersion(versions[1]));
    }

    @BenderTextValue
    public static AppVersionRange fromBenderString(String benderString) {
        return fromIntervalString(benderString);
    }

    @BenderTextValue
    public String asBenderString() {
        return from.map(AppVersion::asBenderString).getOrElse("") + XIVA_PARAMETER_INTERVAL_DELIMITER +
            to.map(AppVersion::asBenderString).getOrElse("");
    }
}
