package ru.yandex.chemodan.util;

import lombok.Data;
import org.junit.Test;

import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.bender.serialize.BenderJsonSerializer;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

public class AppVersionRangeTest {

    private static final BenderJsonParser<JsonWrapper> parser = Bender.jsonParser(JsonWrapper.class);

    private static final BenderJsonSerializer<JsonWrapper> serializerWrapper = Bender.jsonSerializer(JsonWrapper.class);

    @Data
    @BenderBindAllFields
    public static class JsonWrapper {
        private final AppVersionRange range;
    }

    @Test
    public void testParsing() {
        AppVersionRange range = parser.parseJson("{\"range\":\"2.65-3.07\"}").getRange();
        assertTrue(range.isInRangeBothInclusive(new AppVersion("2.66")));
        assertTrue(range.isInRangeBothInclusive(new AppVersion("2.65")));
        assertTrue(range.isInRangeBothInclusive(new AppVersion("3.07")));
        assertFalse(range.isInRangeBothInclusive(new AppVersion("2.64")));
        assertFalse(range.isInRangeBothInclusive(new AppVersion("3.08")));

        AppVersionRange fromRange = parser.parseJson("{\"range\":\"2.65-\"}").getRange();
        assertTrue(fromRange instanceof FromOnlyAppVersionRange);
        assertTrue(fromRange.isInRangeBothInclusive(new AppVersion("2.66")));
        assertTrue(fromRange.isInRangeBothInclusive(new AppVersion("2.65")));
        assertTrue(fromRange.isInRangeBothInclusive(new AppVersion("3.07")));
        assertFalse(fromRange.isInRangeBothInclusive(new AppVersion("2.64")));
        assertTrue(fromRange.isInRangeBothInclusive(new AppVersion("3.08")));

        AppVersionRange toRange = parser.parseJson("{\"range\":\"-3.07\"}").getRange();
        assertTrue(toRange instanceof ToOnlyAppVersionRange);
        assertTrue(toRange.isInRangeBothInclusive(new AppVersion("2.66")));
        assertTrue(toRange.isInRangeBothInclusive(new AppVersion("2.65")));
        assertTrue(toRange.isInRangeBothInclusive(new AppVersion("3.07")));
        assertTrue(toRange.isInRangeBothInclusive(new AppVersion("2.64")));
        assertFalse(toRange.isInRangeBothInclusive(new AppVersion("3.08")));
    }

    @Test
    public void testSerializer() {
        AppVersionRange range = new AppVersionRange(new AppVersion("2.65"), new AppVersion("3.07"));
        assertEquals("{\"range\":\"2.65-3.07\"}", new String(serializerWrapper.serializeJson(new JsonWrapper(range))));

        AppVersionRange fromRange = new FromOnlyAppVersionRange(new AppVersion("2.65"));
        assertEquals("{\"range\":\"2.65-\"}", new String(serializerWrapper.serializeJson(new JsonWrapper(fromRange))));

        AppVersionRange toRange = new ToOnlyAppVersionRange(new AppVersion("3.07"));
        assertEquals("{\"range\":\"-3.07\"}", new String(serializerWrapper.serializeJson(new JsonWrapper(toRange))));
    }
}
