package ru.yandex.chemodan.util;

import org.apache.http.client.HttpClient;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.log.Log4jHelper;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMeta;
import ru.yandex.chemodan.mpfs.MpfsHid;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.digest.Sha256;
import ru.yandex.misc.digest.updatable.UpdatableDigestUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

/**
 * @author akirakozov
 */
public class DigestChecker {
    private static final String MULCA_URL = "http://127.0.0.1:10010/gate";
    private static final String MPFS_HOST = "mpfs-qa.disk.yandex.net";
    private static final String MULCA_SERVICE = "disk";
    private static final String MULCA_NAMESPACE = "disk";

    public static class MpfsFileId {
        private final Either<MpfsHid, Tuple2<PassportUid, String>> id;

        public MpfsFileId(MpfsHid hid) {
            id = Either.left(hid);
        }

        public MpfsFileId(PassportUid uid, String path) {
            id = Either.right(Tuple2.tuple(uid, path));
        }

        public boolean isHid() {
            return id.isLeft();
        }

        public boolean isUidAndPath() {
            return id.isRight();
        }

        public MpfsHid getHid() {
            return id.getLeft();
        }

        public PassportUid getUid() {
            return id.getRight()._1;
        }

        public String getPath() {
            return id.getRight()._2;
        }

        @Override
        public String toString() {
            if (isHid()) {
                return "Hid " + getHid().toString();
            } else {
                return "Uid " + getUid() + ", Path " + getPath();
            }
        }
    }

    private final MulcaClient mulcaClient;
    private final MpfsClient mpfsClient;

    public DigestChecker(String mulcaUrl, String mpfsHost) {
        HttpClient mulcaHttpClient = ApacheHttpClientUtils.singleConnectionClient(Timeout.seconds(60));
        mulcaClient = MulcaClient.custom(mulcaHttpClient, mulcaUrl).withService(MULCA_SERVICE)
                .withNamespace(MULCA_NAMESPACE).build();
        mpfsClient = new MpfsClientImpl(mpfsHost);
    }

    public static void main(String[] args) {
        Log4jHelper.configureTestLogger();

        if (args.length < 1) {
            System.out.println("Enter input file name!");
            return;
        }

        boolean hid = args.length > 1 && args[1].equals("-hid");

        File2 input = new File2(args[0]);
        DigestChecker checker = new DigestChecker(MULCA_URL, MPFS_HOST);

        for (String line : input.readLines()) {
            Option<MpfsFileId> id = parseId(line, hid);
            if (id.isPresent()) {
                checker.checkFileDigests(id.get());
            }
        }
    }

    private void checkFileDigests(MpfsFileId id) {
        try {
            checkFileDigestsUnsafe(id);
        } catch (Exception e) {
            System.out.println(id + ": Error, " + e.getMessage());
        }
    }

    private static Option<MpfsFileId> parseId(String line, boolean hid) {
        try {
            if (hid) {
                return Option.of(new MpfsFileId(new MpfsHid(line)));
            } else {
                String[] parts = line.split(" ");
                return Option.of(
                        new MpfsFileId(PassportUid.cons(Integer.parseInt(parts[0])),
                        UrlUtils.urlDecode(parts[1])));
            }

        } catch (Exception e) {
            System.out.println("Couldn't parse line: " + line + ", error: " + e.getMessage());
        }
        return Option.empty();
    }

    boolean checkFileDigestsUnsafe(MpfsFileId id) {
        MpfsFileInfo info = getFileInfo(id);

        InputStreamSource fileSource =
                mulcaClient.download(MulcaId.fromSerializedString(info.getMeta().getFileMid().get()));
        Tuple2<Md5.Sum, Sha256.Sum> sums = UpdatableDigestUtils.digest(fileSource).asTuple();
        String md5 = sums._1.hex();
        String sha256 = sums._2.hex();

        if (checkDigests(info.getMeta(), md5, sha256)) {
            System.out.println(id + ": Ok");
            return true;
        } else {
            System.out.println(id + ": Incorrect digest. "
                    + "Database md5=" + info.getMeta().getMd5() + ", sha256=" + info.getMeta().getSha256() + ", "
                    + "real md5=" + md5 + ", sha256=" + sha256);
            return false;
        }
    }

    private boolean checkDigests(MpfsFileMeta meta, String md5, String sha256) {
        if (meta.getMd5().isPresent() && meta.getSha256().isPresent()) {
            return md5.equals(meta.getMd5().get()) && sha256.equals(meta.getSha256().get());
        }
        return false;
    }

    private MpfsFileInfo getFileInfo(MpfsFileId id) {
        if (id.isHid()) {
            return mpfsClient.getFileInfoByHid(id.getHid());
        } else {
            return mpfsClient.getFileInfoByUidAndPath(MpfsUser.of(id.getUid()), id.getPath(), Cf.list("file_mid", "md5", "sha256"));
        }
    }
}
