package ru.yandex.chemodan.util;

import java.util.function.Function;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Interval;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class WaitInCycle<V> {
    private volatile Interval interval = null;

    private final Duration cycleDuration;

    private final Duration totalDuration;

    private final Function<V, Boolean> conditionF;

    public WaitInCycle(Duration cycleDuration, Duration totalDuration) {
        this(cycleDuration, totalDuration, v -> true);
    }

    public WaitInCycle(Duration cycleDuration, Duration totalDuration, Function<V, Boolean> conditionF) {
        this.cycleDuration = cycleDuration;
        this.totalDuration = totalDuration;
        this.conditionF = conditionF;
    }

    public WaitInCycle<V> repeatWhile(Function<V, Boolean> conditionF) {
        return new WaitInCycle<>(cycleDuration, totalDuration, conditionF);
    }

    public Interval getInterval() {
        if (interval == null) {
            interval = new Interval(Instant.now(), totalDuration);
        }
        return interval;
    }

    public boolean proceed() {
        return getInterval().contains(Instant.now());
    }

    public Duration elapsed() {
        return interval != null ? new Duration(getInterval().getStart(), Instant.now()) : Duration.ZERO;
    }

    public Duration nextCycleDuration() {
        Duration duration = zeroIfNegative(new Duration(Instant.now(), interval.getEnd()));
        return ComparableUtils.min(cycleDuration, duration);
    }

    public V call(Function<Duration, V> consumer) {
        V result = null;
        boolean consumerCalled = false;
        for(boolean condition = true; condition && proceed(); condition = conditionF.apply(result)) {
            result = consumer.apply(nextCycleDuration());
            consumerCalled = true;
        }
        return consumerCalled ? result : consumer.apply(Duration.ZERO);
    }

    private static Duration zeroIfNegative(Duration duration) {
        return duration.getMillis() > 0 ? duration : Duration.ZERO;
    }
}
