package ru.yandex.chemodan.util.bender;

import lombok.Data;
import org.bson.BsonDocument;
import org.bson.BsonValue;
import org.bson.types.ObjectId;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.mongo3.FiltersX;
import ru.yandex.commune.mongo3.MongoBenderParserSerializer;
import ru.yandex.commune.mongo3.bender.MongoId;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.MoscowTime;

/**
 * @author dbrylev
 */
public class UnderscoreNamesBendingTest {

    @Test
    public void reparse() {
        BenderParserSerializer<ArrayResult> mapper = Bender.cons(ArrayResult.class, configuration());

        ArrayResult array = new ArrayResult(
                Cf.list(new ArrayItem("x", Option.empty()), new ArrayItem("y", Option.of(1))),
                Option.of(new ArrayItem("z", Option.empty())),
                29159, Cf.list(2, 5, 159));

        String serialized = new String(mapper.getSerializer().serializeJson(array));

        Assert.equals("{"
                + "\"item_list\":[{\"string_field\":\"x\"},{\"string_field\":\"y\",\"option_field\":1}],"
                + "\"item_option\":{\"string_field\":\"z\"},"
                + "\"int_field\":29159,\"int_list\":[2,5,159]"
                + "}", serialized);
        Assert.equals(array, mapper.getParser().parseJson(serialized));
    }

    @Test
    public void mongo() {
        MongoBenderParserSerializer<ObjectId, MongoArray> mapper =
                MongoBenderParserSerializer.cons(MongoArray.class, configuration());

        MongoArray array = new MongoArray(
                new MongoArrayId(27, "id"),

                Cf.list(new MongoArrayItem(ObjectId.get(), Option.empty()),
                        new MongoArrayItem(ObjectId.get(), Option.of(MoscowTime.today()))),

                Option.of(new MongoArrayItem(ObjectId.get(), Option.of(MoscowTime.today().plusDays(1)))),
                Cf.list("first", "second", "third"));

        BsonDocument serialized = mapper.serialize(array);

        Assert.equals(array.getId().getFirstColumn(),
                serialized.getDocument(FiltersX.ID_FIELD).getInt32("first_column").getValue());

        Assert.equals(array.getId().getSecondColumn(),
                serialized.getDocument(FiltersX.ID_FIELD).getString("second_column").getValue());

        Assert.equals(array.getItemOption().get().getObjectId(),
                serialized.get("item_option").asDocument().getObjectId("object_id").getValue());

        ListF<BsonDocument> items = Cf.x(serialized.getArray("item_list").getValues()).map(BsonValue::asDocument);
        Assert.isNull(items.first().get("option_date"));

        Assert.equals(
                array.getItemList().last().getOptionDate().get().toDateTimeAtStartOfDay(DateTimeZone.UTC).getMillis(),
                items.last().getDateTime("option_date").getValue());

        Assert.equals(array.getStringList(),
                Cf.x(serialized.getArray("string_list")).map(v -> v.asString().getValue()));

        Assert.equals(array, mapper.parse(serialized));
    }

    private static BenderConfiguration configuration() {
        return UnderscoreNamesBenderConfiguration.extend(BenderConfiguration.defaultConfiguration());
    }

    @Data
    @BenderBindAllFields
    private static class ArrayResult {
        private final ListF<ArrayItem> itemList;
        private final Option<ArrayItem> itemOption;

        private final int intField;
        private final ListF<Integer> intList;
    }

    @Data
    @BenderBindAllFields
    private static class ArrayItem {
        private final String stringField;
        private final Option<Integer> optionField;
    }

    @Data
    @BenderBindAllFields
    private static class MongoArray {
        @MongoId
        private final MongoArrayId id;
        private final ListF<MongoArrayItem> itemList;
        private final Option<MongoArrayItem> itemOption;
        private final ListF<String> stringList;
    }

    @Data
    @BenderBindAllFields
    private static class MongoArrayId {
        private final int firstColumn;
        private final String secondColumn;
    }

    @Data
    @BenderBindAllFields
    private static class MongoArrayItem {
        private final ObjectId objectId;
        private final Option<LocalDate> optionDate;
    }
}
