package ru.yandex.chemodan.util.bookinfo;

import java.io.IOException;
import java.io.Writer;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.log.Log4jHelper;
import ru.yandex.chemodan.util.search.SearchClient;
import ru.yandex.chemodan.util.search.SearchResult;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Vsevolod Tolstopyatov (qwwdfsad)
 */
public class BookInfoGrabber {
    private static final Logger logger = LoggerFactory.getLogger(BookInfoGrabber.class);

    public static final String OUTPUT_FILE_NAME = "book-info.log";

    private final BookInfoFormatter formatter;
    private final SearchClient searchClient;

    public BookInfoGrabber(BookInfoFormatter formatter, SearchClient searchClient) {
        this.formatter = formatter;
        this.searchClient = searchClient;
    }

    public static void main(String[] args) throws IOException {
        Log4jHelper.configureTestLogger();
        Validate.equals(args.length, 2, "Run with file with uids and search-url");
        File2 input = new File2(args[0]);
        ListF<PassportUid> uids = input.readLines().map(Cf.Long.parseF().andThen(PassportUid.consF()));
        BookInfoGrabber grabber = new BookInfoGrabber(new BookInfoFormatter(), new SearchClient(args[1]));
        File2 output = new File2(OUTPUT_FILE_NAME);
        grabber.grab(uids, output);
    }

    public void grab(ListF<PassportUid> uids, File2 output) throws IOException {
        logger.info("Start uids processing");
        try (Writer out = output.asWriterSource().getOutput()) {
            int count = 0;
            for (PassportUid uid : uids) {
                try {
                    ListF<SearchResult.SearchItem> bookInfos = searchClient.getUserBooksInfo(uid);
                    for (SearchResult.SearchItem bookInfo : bookInfos) {
                        out.write(formatter.format(uid, bookInfo));
                        out.write("\n");
                    }
                } catch (Exception e) {
                    logger.warn("Couldn't get books info for uid: " + uid + ", " + e.getMessage(), e);
                }
                if (count % 1000 == 0) {
                    logger.debug("Number of processed uids: " + count);
                    if (output.length() > DataSize.fromGigaBytes(100).toBytes()) {
                        logger.debug("100 GB limit exceeded");
                        System.exit(1);
                    }

                }
                count++;
            }
        }

        logger.info("Uids processing is finished");
    }
}
