package ru.yandex.chemodan.util.date;

import java.math.BigDecimal;

import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Days;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Option;

@Slf4j
@UtilityClass
public class DateTimeUtils {
    public static final long SECONDS_IN_DAY = Days.days(1).toStandardSeconds().getSeconds();
    public static final DateTime MAX_DATE = new DateTime(9999, 1, 1, 0, 0, 0, DateTimeZone.UTC);
    public static final Instant MAX_INSTANT = MAX_DATE.toInstant();

    public static BigDecimal getMillisecondsInMonth(LocalDate month) {
        int daysInMonth = month.dayOfMonth().getMaximumValue();
        return BigDecimal.valueOf(DateTimeUtils.SECONDS_IN_DAY)
                .multiply(BigDecimal.valueOf(1000))
                .multiply(BigDecimal.valueOf(daysInMonth));
    }

    public static boolean isNowInRange(Option<Instant> start, Option<Instant> end) {
        Instant now = Instant.now();
        return isInRange(start, end, now);
    }

    public static boolean isInRange(Option<Instant> start, Option<Instant> end, Instant instant) {
        Instant from = start.orElse(instant.minus(1));
        Instant to = end.orElse(instant.plus(1));

        return (from.isEqual(instant) || from.isBefore(instant))
                && (to.isEqual(instant) || to.isAfter(instant));
    }

    public static boolean isNowInRange(Instant from, Instant to) {
        return isNowInRange(Option.of(from), Option.of(to));
    }

    public static boolean isNowInRange(Instant from, Option<Instant> to) {
        return isNowInRange(Option.of(from), to);
    }

    public static Instant floorTimeForTimezone(Instant dateTime, DateTimeZone timeZone) {
        return new Instant(new LocalDate(dateTime, timeZone).toDateTimeAtStartOfDay(timeZone));
    }

    public static Instant ceilTimeForTimezone(Instant dateTime, DateTimeZone timeZone) {
        Instant flooredTime = floorTimeForTimezone(dateTime, timeZone);
        if (flooredTime.equals(dateTime)) {
            return dateTime;
        }

        return flooredTime.plus(Duration.standardDays(1));
    }

    public static DateTime getNextMonthFirstDate(Instant date) {
        return date.toDateTime().plusMonths(1).withDayOfMonth(1).withTime(0, 0, 0, 0);
    }

    public static DateTime getMonthFirstDate(Instant date) {
        return date.toDateTime().withDayOfMonth(1).withTime(0, 0, 0, 0);
    }

    public static boolean areClose(Instant date1, Instant date2, Duration epsilon) {
        if (date1.equals(date2)) {
            return true;
        }

        if (date1.isAfter(date2)) {
            return date1.minus(epsilon).isBefore(date2);
        }

        return date1.plus(epsilon).isBefore(date2);
    }

    public static Instant toInstant(LocalDate date) {
        return date.toDateTimeAtStartOfDay().toInstant();
    }

    public static Instant yesterday() {
        return Instant.now().minus(Duration.standardDays(1));
    }

    public static LocalDate yesterdayLd() {
        return yesterday().toDateTime().toLocalDate();
    }
}
