package ru.yandex.chemodan.util.date;

import javax.annotation.Nonnull;

import lombok.Getter;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.IterableF;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.impl.AbstractPrefetchingIterator;
import ru.yandex.chemodan.util.yt.DateRangeTablePath;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class LocalDateRange extends DefaultObject implements IterableF<LocalDate> {
    @Getter
    private final LocalDate start;

    @Getter
    private final LocalDate end;

    public LocalDateRange(LocalDate start, LocalDate end) {
        this.start = start;
        this.end = end;
    }

    public static LocalDateRange daysUpTo(int days, LocalDate end) {
        return new LocalDateRange(end.minusDays(days), end);
    }

    public static Option<LocalDateRange> parseSafe(String str) {
        String[] chunks = str.split("_");
        if (chunks.length != 2) {
            return Option.empty();
        }
        try {
            return Option.of(parse(chunks[0], chunks[1]));
        } catch (IllegalArgumentException e) {
            return Option.empty();
        }
    }

    public static LocalDateRange parse(String startStr, String endStr) {
        return new LocalDateRange(LocalDate.parse(startStr), LocalDate.parse(endStr));
    }

    @Nonnull
    @Override
    public IteratorF<LocalDate> iterator() {
        return new AbstractPrefetchingIterator<LocalDate>() {
            LocalDate next = start;

            @Override
            protected Option<LocalDate> fetchNext() {
                if (next.isAfter(end)) {
                    return Option.empty();
                }
                Option<LocalDate> result = Option.of(next);
                next = next.plusDays(1);
                return result;
            }
        };
    }

    @Override
    public String toString() {
        return String.format("%s_%s", start, end);
    }

    public boolean overlaps(LocalDateRange other) {
        return !independentOf(other);
    }

    public boolean independentOf(LocalDateRange other) {
        return this.end.isBefore(other.start) || this.start.isAfter(other.end);
    }

    public boolean subsetOf(LocalDateRange other) {
        return !notSubsetOf(other);
    }

    private boolean notSubsetOf(LocalDateRange other) {
        return this.start.isBefore(other.start) || this.end.isAfter(other.end);
    }

    public boolean includes(LocalDateRange other) {
        return other.subsetOf(this);
    }

    public SetF<LocalDate> toSet() {
        return iterator()
                .toSet();
    }

    public static SetF<LocalDate> collectLocalDateSet(CollectionF<DateRangeTablePath> existingRanges) {
        return existingRanges.map(DateRangeTablePath::getDateRange)
                .map(LocalDateRange::toSet)
                .reduceLeftO(SetF::plus)
                .getOrElse(Cf.set());
    }
}
