package ru.yandex.chemodan.util.http;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.net.SocketTimeoutException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;

import javax.net.ssl.SSLException;

import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.handler.ssl.SslContextBuilder;
import org.apache.http.conn.ConnectTimeoutException;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.asynchttpclient.Request;
import org.asynchttpclient.Response;

import ru.yandex.bolts.collection.Unit;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.asyncHttpClient2.AsyncHttpClientUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.devnull.DevNullOutputStream;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;

/**
 * @author tolmalev
 */
public class AsyncHttpClientWithRetries implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(AsyncHttpClientWithRetries.class);

    private final AsyncHttpClient asyncHttpClient;
    private final int retriesCount;

    public AsyncHttpClientWithRetries(Timeout timeout, int maxConnectionsCount, int retriesCount, String userAgent) {
        this.retriesCount = retriesCount;

        // XXX: is 4 enough
        NioEventLoopGroup eventLoopGroup =
                new NioEventLoopGroup(4, new ThreadNameIndexThreadFactory(userAgent + "-worker-"));

        try {
            this.asyncHttpClient = new DefaultAsyncHttpClient(new DefaultAsyncHttpClientConfig.Builder()
                    .setEventLoopGroup(eventLoopGroup)
                    .setUserAgent(userAgent)

                    .setConnectTimeout((int) timeout.getConnectTimeoutMilliseconds())
                    .setSslSessionTimeout((int) timeout.getConnectTimeoutMilliseconds())

                    .setRequestTimeout((int) timeout.getSoTimeoutMilliseconds())
                    .setReadTimeout((int) timeout.getSoTimeoutMilliseconds())

                    .setSslContext(SslContextBuilder.forClient().build())

                    .setMaxConnections(maxConnectionsCount)
                    .build()
            );
        } catch (SSLException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public CompletableFuture<InputStreamSource> executeHttpGet(String url) {
        return queryStream(asyncHttpClient.prepareGet(url).build());
    }

    public CompletableFuture<Unit> queryIgnoreResponse(Request request) {
        return queryStream(request).thenApply(iss -> {
            InputStream stream = iss.getInputUnchecked();
            IoUtils.copy(stream, new DevNullOutputStream());
            IoUtils.closeQuietly(stream);
            return Unit.U;
        });
    }

    public CompletableFuture<InputStreamSource> queryStream(Request request) {
        logger.info("{} url: {}", request.getMethod(), request.getUrl());

        return retry(request, retriesCount)
                .handle((r, e) -> {
                    CompletableFuture<InputStreamSource> f = new CompletableFuture<>();
                    if (e != null) {
                        f.completeExceptionally(e);
                    } else {
                        if (HttpStatus.is2xx(r.getStatusCode())) {
                            f.complete(InputStreamSourceUtils.wrap(r.getResponseBodyAsStream()));
                        } else {
                            IoUtils.closeQuietly(r.getResponseBodyAsStream());
                            f.completeExceptionally(
                                    new RuntimeException("Failed to execute request. Code: " + r.getStatusCode()));
                        }
                    }
                    return f;
                })
                .thenCompose(x -> x);
    }

    private CompletableFuture<Response> retry(Request request, int retriesLeft) {
        ThreadLocalTimeout.check();

        CompletableFuture<Response> currentTryFuture = AsyncHttpClientUtils.execute(asyncHttpClient, request);
        if (retriesLeft == 0) {
            return currentTryFuture;
        }

        return currentTryFuture.handle((r, e) -> {
            // response received but it's 5xx
            if (e == null && !HttpStatus.is5xx(r.getStatusCode())) {
                return CompletableFuture.completedFuture(r);
            }
            // if there is an exception but not timeout
            if (e != null && !(
                        e instanceof SocketTimeoutException
                        || e instanceof ConnectTimeoutException
                        || e instanceof TimeoutException)
                    )
            {
                CompletableFuture<Response> f = new CompletableFuture<>();
                f.completeExceptionally(e);
                return f;
            }

            return retry(request, retriesLeft - 1);
        }).thenCompose(x -> x);
    }

    @Override
    public void close() throws IOException {
        asyncHttpClient.close();
    }
}
