package ru.yandex.chemodan.util.http;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;

import com.xebialabs.restito.server.StubServer;
import org.glassfish.grizzly.http.Method;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.util.test.StubServerUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.ip.IpPortUtils;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;

import static com.xebialabs.restito.builder.stub.StubHttp.whenHttp;
import static com.xebialabs.restito.semantics.Action.custom;
import static com.xebialabs.restito.semantics.Action.stringContent;
import static com.xebialabs.restito.semantics.Condition.method;

/**
 * @author tolmalev
 */
public class AsyncHttpClientWithRetriesTest {
    AsyncHttpClientWithRetries client = new AsyncHttpClientWithRetries(Timeout.milliseconds(100), 10, 3, "tests");

    @Test
    public void retry5xx() {
        StubServerUtils.withStubServer((port, stubServer) -> {
            AtomicInteger i = new AtomicInteger();
            whenHttp(stubServer).match(method(Method.GET)).then(custom(r -> {
                if (i.incrementAndGet() == 3) {
                    r.setStatus(200);
                    stringContent("mu-ha-ha").apply(r);
                } else {
                    r.setStatus(500);
                }
                return r;
            }));


            String text = client.executeHttpGet(url(port)).get().readText();
            Assert.equals("mu-ha-ha", text);
            Assert.equals(3, i.get(), "Executed not 3 retries");
        });
    }

    @Test
    public void retryTimeout() {
        StubServerUtils.withStubServer((port, stubServer) -> {
            AtomicInteger i = new AtomicInteger();
            whenHttp(stubServer).match(method(Method.GET)).then(custom(r -> {
                if (i.incrementAndGet() == 3) {
                    r.setStatus(200);
                    stringContent("mu-ha-ha").apply(r);
                } else {
                    ThreadUtils.sleep(1000);
                    r.setStatus(200);
                    stringContent("mu-ha-ha-timeout").apply(r);
                }
                return r;
            }));


            String text = client.executeHttpGet(url(port)).get().readText();
            Assert.equals("mu-ha-ha", text);
            Assert.equals(3, i.get(), "Executed not 3 retries");
        });
    }

    @Test
    @Ignore // Takes much more then 1 second at CI
    public void manyRequestsCompletesInOneSecond() throws ExecutionException, InterruptedException {
        client = new AsyncHttpClientWithRetries(Timeout.milliseconds(2000), 100, 3, "tests");

        ListF<Integer> ports = Cf.range(0, 100).map(i -> IpPortUtils.getFreeLocalPort().getPort());
        ListF<StubServer> servers = ports.map(port -> {
            StubServer server = new StubServer(port).run();

            whenHttp(server).match(method(Method.GET)).then(custom(r -> {
                ThreadUtils.sleep(1000);

                r.setStatus(200);
                stringContent("mu-ha-ha").apply(r);
                return r;
            }));

            return server;
        });

        Instant start = Instant.now();

        ListF<CompletableFuture<InputStreamSource>> futures =
                ports.map(port -> client.executeHttpGet("http://localhost:" + (port)));

        Duration passedToInit = new Duration(start, Instant.now());
        Assert.le(passedToInit.getMillis(), 1000L);

        for (CompletableFuture<InputStreamSource> future : futures) {
            String text = future.get().readText();
            Assert.equals("mu-ha-ha", text);
        }

        Duration passed = new Duration(start, Instant.now());
        Assert.le(passed.getMillis(), 5000L);

        servers.forEach(StubServer::stop);
    }

    @Test
    public void https() throws ExecutionException, InterruptedException {
        client = new AsyncHttpClientWithRetries(Timeout.milliseconds(10000), 10, 3, "tests");
        client.executeHttpGet("https://www.yandex.ru").get().readText();
    }

    private static String url(int port) {
        return "http://localhost:" + port;
    }
}
