package ru.yandex.chemodan.util.http;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.Args;
import org.apache.http.util.EntityUtils;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.io.http.apache.v4.InstrumentedCloseableHttpClient;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.monica.util.measure.MeasureInfo;

/**
 * @author tolmalev
 */
public class DiskInstrumentedClosableHttpClient extends InstrumentedCloseableHttpClient {
    private final Log log = LogFactory.getLog(getClass());

    private final InstrumentMap invocationsRequest;
    private final InstrumentMap invocationsTotal;

    private final MeterMap failByReason;

    public DiskInstrumentedClosableHttpClient(CloseableHttpClient client, RequestConfig defaultRequestConfig, InstrumentMap baseInvocations,
          InstrumentMap invocationsRequest, InstrumentMap invocationsTotal, MeterMap failByReason)
    {
        super(client, defaultRequestConfig, baseInvocations);
        this.failByReason = failByReason;
        this.invocationsRequest = invocationsRequest;
        this.invocationsTotal = invocationsTotal;
    }

    // Copy paste from CloseableHttpClient with success instrumentation
    @Override
    public <T> T execute(final HttpHost target, final HttpRequest request,
                         final ResponseHandler<? extends T> responseHandler, final HttpContext context)
            throws IOException, ClientProtocolException
    {
        Args.notNull(responseHandler, "Response handler");

        Instant start = Instant.now();
        final CloseableHttpResponse response;
        try {
            response = execute(target, request, context);
        } catch (Exception e) {
            invocationsRequest.update(new MeasureInfo(new Duration(start, Instant.now()), false), new MetricName(Cf.list()), UpdateMode.RECURSIVE);
            invocationsTotal.update(new MeasureInfo(new Duration(start, Instant.now()), false), new MetricName(Cf.list()), UpdateMode.RECURSIVE);
            failByReason.inc(new MetricName(Cf.list("request", e.getClass().getSimpleName())), UpdateMode.RECURSIVE);
            throw e;
        }

        int statusCode = response.getStatusLine().getStatusCode();
        MetricName metricName = new MetricName(Cf.list("status_" + statusCode));
        invocationsRequest.update(new MeasureInfo(new Duration(start, Instant.now()), true), metricName, UpdateMode.RECURSIVE);

        try {
            final T result;
            try {
                result = responseHandler.handleResponse(response);
            } catch (Exception e) {
                Duration elapsed = new Duration(start, Instant.now());
                failByReason.inc(new MetricName(Cf.list(
                        "status_" + statusCode,
                        "handler",
                        e.getClass().getSimpleName())), UpdateMode.RECURSIVE);
                invocationsTotal.update(new MeasureInfo(elapsed, false), metricName, UpdateMode.RECURSIVE);
                throw e;
            }

            Duration elapsed = new Duration(start, Instant.now());
            invocationsTotal.update(new MeasureInfo(elapsed, true), metricName, UpdateMode.RECURSIVE);

            final HttpEntity entity = response.getEntity();
            EntityUtils.consume(entity);
            return result;
        } catch (final ClientProtocolException t) {
            // Try to salvage the underlying connection in case of a protocol exception
            final HttpEntity entity = response.getEntity();
            try {
                EntityUtils.consume(entity);
            } catch (final Exception t2) {
                // Log this exception. The original exception is more
                // important and will be thrown to the caller.
                this.log.warn("Error consuming content after an exception.", t2);
            }
            throw t;
        } finally {
            response.close();
        }
    }
}
