package ru.yandex.chemodan.util.http;

import java.io.IOException;
import java.util.Objects;
import java.util.Optional;

import lombok.Data;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.impl.DefaultConnectionReuseStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.pool.PoolStats;
import org.apache.http.protocol.HttpContext;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.boot.value.OverridableValue;
import ru.yandex.chemodan.boot.value.OverridableValuePrefixAware;
import ru.yandex.chemodan.http.CommonHeaders;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.tvm2.SingleClientResolver;
import ru.yandex.inside.passport.tvm2.AddServiceTicketInterceptor;
import ru.yandex.inside.passport.tvm2.AddUserTicketInterceptor;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.AddRequestAttemptHeaderInterceptor;
import ru.yandex.misc.io.http.apache.CompoundRetryStrategy;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.parse.CommaSeparated;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.version.AppName;
import ru.yandex.misc.version.Version;

/**
 * @author tolmalev
 */
@Data
public class HttpClientConfigurator implements MonicaContainer, OverridableValuePrefixAware {
    @Autowired
    private Optional<AppName> appName;
    @Autowired
    private Optional<Version> version;

    @OverridableValue("${http.user.agent}")
    private String userAgentStr;

    @OverridableValue("${http.connect.timeout}")
    private Duration timeoutConnection;

    @OverridableValue("${http.socket.timeout}")
    private CommaSeparated timeoutSocket;

    @OverridableValue("${http.socket.keep-alive}")
    private boolean keepAliveSocket;

    @OverridableValue("${http.max.connections}")
    private int maxConnections;

    @OverridableValue("${http.retry.count}")
    private int retryCount;
    @OverridableValue("${http.retry.if-sent}")
    private boolean retryIfSent;

    @OverridableValue("${http.retry.max_budget}")
    private int retryMaxBudget;
    @OverridableValue("${http.retry.add_per_requests}")
    private int addRetryRequests;

    @OverridableValue("${http.retry.interval}")
    private Duration retryInterval;
    @OverridableValue("${http.retry.strategy}")
    private CompoundRetryStrategy.RetryDelayStrategy retryDelayStrategy;
    @OverridableValue("${http.retry.reconnect}")
    private boolean reconnectOnRetry;

    @OverridableValue("${http.tvm.disabled:-false}")
    private boolean tvmDisabled;
    @OverridableValue("${http.tvm.force_dst_id:-0}")
    private int tvmDstId;

    @Autowired
    private Optional<AddServiceTicketInterceptor> addServiceTicketInterceptor;

    @Autowired
    private Optional<AddUserTicketInterceptor> addUserTicketInterceptor;

    @Autowired
    private Optional<Tvm2> tvm2;

    @MonicaMetric
    private final MeterMap retryByReason = new MeterMap();
    @MonicaMetric
    private final MeterMap retryByNum = new MeterMap();
    @MonicaMetric
    private final MeterMap failByReason = new MeterMap();
    @MonicaMetric
    private final InstrumentMap invocations = new InstrumentMap();
    @MonicaMetric
    private final InstrumentMap invocationsRequest = new InstrumentMap();

    private String overridableValuePrefix;

    private Option<DiskInstrumentedClosableHttpClient> lastConfigured = Option.empty();
    private Option<HttpClientConnectionManager> connectionManager = Option.empty();

    public HttpClient configure() {
        return createBuilder().build();
    }

    public HttpClient withCustomTimeout(Timeout timeout) {
        return createBuilder().withTimeout(timeout).build();
    }

    public ApacheHttpClientUtils.Builder createBuilder() {
        ApacheHttpClientUtils.Builder builder = new Builder()
                .withTimeout(new Timeout(getSocketTimeout(), timeoutConnection.getMillis()))
                .withSocketConfig(SocketConfig.custom().setSoKeepAlive(keepAliveSocket).build())
                .withHttpsSupport(HttpsUtils.httpsSupport())
                .withUserAgent(getUserAgent());

        if (maxConnections > 1) {
            builder = builder.multiThreaded().withMaxConnections(maxConnections);
        } else {
            builder = builder.singleThreaded();
        }
        if (retryCount > 0) {
            CompoundRetryStrategy strategy = new CompoundRetryStrategy(retryCount, retryIfSent, retryMaxBudget, addRetryRequests, retryInterval,
                    retryDelayStrategy, Cf.set(507));
            strategy.enableMonitoring(retryByReason, retryByNum);

            builder = builder.withRequestRetryHandler(strategy);
            builder = builder.withServiceUnavailableRetryStrategy(strategy);
        }

        builder.withInterceptorLast(new HttpRequestInterceptor() {
            @Override
            public void process(HttpRequest request, HttpContext context) {
                if (request.getFirstHeader(CommonHeaders.YANDEX_CLOUD_REQUEST_ID) == null) {
                    YandexCloudRequestIdHolder.getO().forEach(
                            v -> request.addHeader(CommonHeaders.YANDEX_CLOUD_REQUEST_ID, v));
                }
            }
        });
        builder.withInterceptorLast(new AddRequestAttemptHeaderInterceptor());
        configureSocketTimeoutInterceptor(builder);

        if (reconnectOnRetry) {
            builder.withConnectionReuseStrategy(new DefaultConnectionReuseStrategy() {
                @Override
                public boolean keepAlive(HttpResponse response, HttpContext context) {
                    int statusCode = response.getStatusLine().getStatusCode();
                    if (statusCode != 507 && HttpStatus.is5xx(statusCode)) {
                        return false;
                    }
                    return super.keepAlive(response, context);
                }
            });
        }

        if (!tvmDisabled) {
            if (tvmDstId > 0) {
                if (!tvm2.isPresent()) {
                    throw new IllegalArgumentException("tvm dst configured : "
                            + tvmDstId
                            + ", but no tvm2 in context for http service " + overridableValuePrefix
                    );
                }
                tvm2.get().addDstClientIds(Cf.list(tvmDstId));
                builder.withInterceptorLast(tvm2.get().serviceTicketInterceptor(new SingleClientResolver(tvmDstId)));
            } else {
                addServiceTicketInterceptor.filter(Objects::nonNull).map(builder::withInterceptorLast);
            }
            addUserTicketInterceptor.filter(Objects::nonNull).map(builder::withInterceptorLast);
        }

        return builder;
    }

    private void configureSocketTimeoutInterceptor(ApacheHttpClientUtils.Builder builder) {
        ListF<Duration> timeouts = getSocketTimeoutList();
        builder.withInterceptorLast(new HttpRequestInterceptor() {
            @Override
            public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {
                if (timeouts.size() == 1) {
//                    Аналогичное речение привело к проблемам в связке мпфс-ксива
//                    request.setHeader("X-Request-Timeout", Long.toString(timeouts.first().getMillis()));
                    return;
                }

                Option<Integer> attempt = Option.ofNullable(context.getAttribute(AddRequestAttemptHeaderInterceptor.REQUEST_ATTEMPT_ATTRIBUTE))
                        .map(Object::toString)
                        .flatMapO(Cf.Integer::parseSafe);

                Duration timeout;
                if (!attempt.isPresent()) {
                    timeout = timeouts.first();
                } else if (attempt.get() - 1 < timeouts.size()) {
                    timeout = timeouts.get(attempt.get() - 1);
                } else {
                    timeout = timeouts.last();
                }

//                Аналогичное речение привело к проблемам в связке мпфс-ксива
//                request.setHeader("X-Request-Timeout", Long.toString(timeout.getMillis()));

                if (attempt.isPresent() && attempt.get() > 1 && context instanceof HttpClientContext) {

                    HttpClientContext clientContext = (HttpClientContext) context;
                    clientContext.setRequestConfig(RequestConfig
                            .copy(clientContext.getRequestConfig())
                            .setSocketTimeout((int) timeout.getMillis())
                            .build()
                    );
                }
            }
        });
    }

    public String getUserAgent() {
        Optional<String> userAgent = Optional.ofNullable(StringUtils.trimToNull(userAgentStr));

        if (userAgent.isPresent()) {
            return userAgent.get();
        }

        String shortHost = HostnameUtils.localHostname().split("\\.")[0];
        userAgent = appName
                .map(AppName::appName)
                .map(app -> version.map(ver -> shortHost + "/" + app + "/" + ver.getProjectVersion()).orElse(app));

        return userAgent.get();
    }

    public long getSocketTimeout() {
        return getSocketTimeoutList().first().getMillis();
    }

    private ListF<Duration> getSocketTimeoutList() {
        return timeoutSocket.list.map(TimeUtils::toDuration);
    }

    public long getConnectionTimeout() {
        return timeoutConnection.getMillis();
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "httpconfigurator",
                new MetricName("httpconfigurator", Option.ofNullable(overridableValuePrefix).getOrElse(instanceName)),
                "Metrics from http configurator for " + Option.ofNullable(overridableValuePrefix).getOrElse(instanceName)
        );
    }

    @MonicaMetric
    public int connectionsAvailable_ammx() {
        return getPoolStatsO()
                .map(PoolStats::getAvailable)
                .getOrElse(0);
    }

    @MonicaMetric
    public int connectionsLeased_ammx() {
        return getPoolStatsO()
                .map(PoolStats::getLeased)
                .getOrElse(0);
    }

    @MonicaMetric
    public int connectionsMax_ammx() {
        return getPoolStatsO()
                .map(PoolStats::getMax)
                .getOrElse(0);
    }

    @MonicaMetric
    public int connectionsPending_ammx() {
        return getPoolStatsO()
                .map(PoolStats::getPending)
                .getOrElse(0);
    }

    private Option<PoolStats> getPoolStatsO() {
        return connectionManager
                .filter(cm -> cm instanceof PoolingHttpClientConnectionManager)
                .cast(PoolingHttpClientConnectionManager.class)
                .map(PoolingHttpClientConnectionManager::getTotalStats);
    }

    @Override
    public void setOverridableValuePrefix(String prefix) {
        this.overridableValuePrefix = prefix;
    }

    private class Builder extends ApacheHttpClientUtils.Builder {
        public CloseableHttpClient build() {
            if (!connectionManagerSet) {
                httpClientBuilder.setConnectionManager(createDefaultConnectionManager());
            }
            if (disableRedirectHandling) {
                httpClientBuilder.disableRedirectHandling();
            }

            RequestConfig config = requestConfigBuilder.build();
            CloseableHttpClient httpClient = httpClientBuilder
                    .setDefaultRequestConfig(config)
                    .build();


            DiskInstrumentedClosableHttpClient client = new DiskInstrumentedClosableHttpClient(httpClient, config, ApacheHttpClientUtils.invocations,
                    invocationsRequest, invocations, failByReason);

            lastConfigured = Option.of(client);
            return client;
        }

        @Override
        protected HttpClientConnectionManager createDefaultConnectionManager() {
            connectionManager = Option.of(super.createDefaultConnectionManager());
            return connectionManager.get();
        }
    }
}
