package ru.yandex.chemodan.util.http;

import java.io.IOException;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.FileEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.ContentBody;
import org.apache.http.entity.mime.content.StringBody;
import org.dom4j.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.http.CommonHeaders;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.IoFunction;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadRootDom4jElementResponseHandler;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * Not really general-purpose.
 *
 * @author vavinov
 */
public class HttpClientUtils {
    public static final Logger logger = LoggerFactory.getLogger(HttpClientUtils.class);

    public static HttpGet httpGetRanged(String uri, long from) {
        HttpGet get = new HttpGet(uri);
        get.setHeader("Range", "bytes=" + from + "-");
        return get;
    }

    public static HttpPut httpPut(String uri, byte[] data, MapF<String, String> headers) {
        return httpPut(uri, data, headers, false);
    }

    public static HttpPut httpPut(String uri, byte[] data,
            MapF<String, String> headers, boolean chunked)
    {
        HttpPut put = new HttpPut(uri);
        AbstractHttpEntity entity = new ByteArrayEntity(data);
        entity.setChunked(chunked);
        put.setEntity(entity);
        for (Tuple2<String, String> t : headers.entries()) {
            put.setHeader(t._1, t._2);
        }
        return put;
    }

    public static int downloadStatus(HttpClient client, HttpUriRequest request) {
        try {
            return client.execute(request, response -> {
                for (HttpEntity entity : Option.ofNullable(response.getEntity())) {
                    entity.getContent().close();
                }
                return response.getStatusLine().getStatusCode();
            });
        } catch (ClientProtocolException ex) {
            throw ExceptionUtils.translate(ex);
        } catch (IOException ex) {
            throw IoUtils.translate(ex);
        }
    }

    public static Element parseXmlResponse(HttpClient client, HttpUriRequest request) {
        Element el = ApacheHttpClientUtils.execute(request, client,
                new ReadRootDom4jElementResponseHandler());
        //logger.info(request.getRequestLine() + "\n" + el.asXML());
        return el;
    }

    public static MapF<String, ListF<String>> parseResponseHeaders(
            HttpClient client, HttpUriRequest request)
            throws Exception
    {
        return client.execute(request, response -> {
            if (response.getStatusLine().getStatusCode() >= 300) {
                throw new RuntimeException(response.getStatusLine().toString());
            }
            MapF<String, ListF<String>> headers = Cf.hashMap();
            for (Header h : response.getAllHeaders()) {
                headers.getOrElseUpdate(h.getName(), Cf.<String>arrayList()).add(h.getValue());
            }
            return headers;
        });
    }

    public static HttpPut httpPut(String uri, byte[] data) {
        return httpPut(uri, data, Cf.map());
    }

    public static HttpPut httpPutZipped(String uri, byte[] data) {
        return httpPut(uri, data, Cf.map(HttpHeaders.CONTENT_ENCODING, "gzip"), true);
    }

    public static HttpPost httpPost(String uri, MapF<String, ?> args) {
        HttpPost post = ApacheHttpClientUtils.prepareHttpPost(UrlUtils.uri(uri), args);
        post.setHeader(CommonHeaders.YANDEX_CLOUD_REQUEST_ID, "tst-" + Random2.R.nextAlnum(6));
        return post;
    }

    public static HttpPost httpPostMultipart(String uri, String partName, String partContent) {
        return httpPostMultipart(uri, Cf.map(partName, consStringBodyF.apply(partContent)),
                HttpMultipartMode.BROWSER_COMPATIBLE);
    }

    public static HttpPost httpPostMultipart(String uri, MapF<String, String> args,
            String streamPartName, ContentBody streamBody)
    {
        return httpPostMultipart(uri, args.mapValues(consStringBodyF)
                .plus1(streamPartName, streamBody),
                HttpMultipartMode.BROWSER_COMPATIBLE);
    }

    public static HttpPost httpPostMultipart(String uri, MapF<String, String> args) {
        return httpPostMultipart(uri, args.mapValues(consStringBodyF), HttpMultipartMode.BROWSER_COMPATIBLE);
    }

    public static HttpPost httpPostFile(String uri, File2 file) {
        try {
            HttpPost post = new HttpPost(uri);
            FileEntity entity = new FileEntity(file.getFile(), ContentType.create("image/jpeg"));
            post.addHeader(entity.getContentType());
            post.setEntity(entity);
            return post;
        } catch (Exception ex) {
            throw ExceptionUtils.translate(ex);
        }
    }

    private static final Function<String, ContentBody> consStringBodyF = (IoFunction<String, ContentBody>) s -> new StringBody(s);

    private static HttpPost httpPostMultipart(String uri, MapF<String, ContentBody> parts,
            HttpMultipartMode mode)
    {
        try {
            HttpPost post = new HttpPost(uri);
            MultipartEntity entity = new MultipartEntity(mode);
            for (String name : parts.keys()) {
                entity.addPart(name, parts.getTs(name));
            }
            //post.addHeader(HttpHeaders.EXPECT, HTTP.EXPECT_CONTINUE); // no server response otherwise O_o
            post.addHeader(entity.getContentType());
            post.setEntity(entity);
            return post;
        } catch (Exception ex) {
            throw ExceptionUtils.translate(ex);
        }
    }
}
