package ru.yandex.chemodan.util.http;

import java.io.IOException;

import org.apache.http.HttpResponse;
import org.apache.http.client.ResponseHandler;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.HttpStatus;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class HttpExceptionResponseHandler<T> implements ResponseHandler<T> {
    private final ResponseHandler<T> responseHandler;

    private final SetF<HttpStatus.StatusClass> okStatusClasses;

    private final SetF<Integer> okStatusCodes;

    public HttpExceptionResponseHandler(
            ResponseHandler<T> responseHandler,
            SetF<HttpStatus.StatusClass> okStatusClasses,
            SetF<Integer> okStatusCodes)
    {
        this.responseHandler = responseHandler;
        this.okStatusClasses = okStatusClasses;
        this.okStatusCodes = okStatusCodes;
    }

    public static <T> Builder<T> build(ResponseHandler<T> responseHandler) {
        return new Builder<>(responseHandler);
    }

    @Override
    public T handleResponse(HttpResponse response) {
        if (!isOkStatus(response.getStatusLine().getStatusCode())) {
            throw new HttpException(response.getStatusLine().getStatusCode());
        }

        try {
            return responseHandler.handleResponse(response);
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public boolean isOkStatus(int statusCode) {
        return okStatusCodes.containsTs(statusCode) || okStatusClasses.exists(cls -> cls.matches(statusCode));
    }

    public static class Builder<T> {
        private final ResponseHandler<T> responseHandler;

        private final SetF<HttpStatus.StatusClass> okStatusClasses = Cf.hashSet(HttpStatus.StatusClass.SC2XX);

        private final SetF<Integer> okStatusCodes = Cf.hashSet();

        private Builder(ResponseHandler<T> responseHandler) {
            this.responseHandler = responseHandler;
        }

        public Builder addOkStatuses(int... okStatusCodes) {
            this.okStatusCodes.addAll(Cf.intList(okStatusCodes));
            return this;
        }

        public HttpExceptionResponseHandler<T> build() {
            return new HttpExceptionResponseHandler<>(responseHandler, okStatusClasses, okStatusCodes);
        }
    }
}
