package ru.yandex.chemodan.util.http;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.util.EntityUtils;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 *
 * RecoverableInputStream is required for downloading big files from the internet using ranged get requests and
 * auto recover stream connection skipping already streamed bytes if an error occures. It also allows to specify
 * number of retries for such reconnect.
 *
 * Note: it will not work if target service doesn't support ranged get request.
 */
public class RecoverableInputStream extends InputStream {
    private static final Logger logger = LoggerFactory.getLogger(RecoverableInputStream.class);

    private HttpClient httpClient;
    private String uri;
    private long currentOffset = 0;
    private int retryRemains;
    private Option<InputStream> currentInputStream = Option.empty();

    public RecoverableInputStream(HttpClient httpClient, String uri, int retryRemains) {
        this.httpClient = httpClient;
        this.uri = uri;
        this.retryRemains = retryRemains;
    }

    @Override
    public int read(byte[] buf) throws IOException {
        try {
            if (!currentInputStream.isPresent()) {
                updateCurrentInputStream();
            }
            int read = currentInputStream.get().read(buf);

            currentOffset += read;
            return read;
        } catch (IOException e) {
            logger.info("Unable to read from RecoverableInputStream", e);
            retryRemains--;
            currentInputStream.get().close();
            currentInputStream = Option.empty();
            updateCurrentInputStream();
        }
        return 0;
    }

    @Override
    public int read() throws IOException {
        return 0;
    }

    private void updateCurrentInputStream() {
        String describe = toString();
        logger.debug("Stream file: " + describe);

        if (retryRemains > 0) {
            HttpEntity entity = null;
            HttpGet httpGet = HttpClientUtils.httpGetRanged(uri, currentOffset);
            try {
                HttpResponse httpResponse = httpClient.execute(httpGet);
                int statusCode = httpResponse.getStatusLine().getStatusCode();
                entity = httpResponse.getEntity();
                if (!HttpStatus.is2xx(statusCode)) {
                    String error = String.format("Streaming file '%s' error. Server returned %s",
                            describe, httpResponse.getStatusLine().getReasonPhrase());
                    throw new HttpException(statusCode, error);
                }
                currentInputStream = Option.of(new EntityInputStream(entity));
            } catch (Exception e) {
                if (e instanceof HttpException && HttpStatus.is4xx(((HttpException) e).getStatusCode().getOrElse(0))) {
                    throw ExceptionUtils.translate(e);
                } else {
                    String message = String.format("Unable to resume streaming file by url %s with offset %s. Retrying",
                            uri, currentOffset);
                    logger.info(message, e);
                    currentInputStream = Option.empty();
                    retryRemains--;
                    updateCurrentInputStream();
                }
            } finally {
                if (!currentInputStream.isPresent()) {
                    try {
                        EntityUtils.consume(entity);
                    } catch (IOException e) {
                        logger.info("Unable to consume httpEntity: ", e);
                    }
                }
            }
        } else {
            throw new RuntimeIoException("Streaming file by url " + uri + " failed because reached max number of retries");
        }
    }

    @Override
    public void close() {
        if (currentInputStream.isPresent()) {
            try {
                currentInputStream.get().close();
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        }
    }

    @Override
    public String toString() {
        return "RecoverableInputStream [" +
                "uri=" + uri + ", " +
                "readBytes=" + currentOffset + ", " +
                "remainingAttempts=" + retryRemains + "]";
    }

    private static final class EntityInputStream extends FilterInputStream {
        private final HttpEntity httpEntity;

        private EntityInputStream(HttpEntity httpEntity) throws IOException {
            super(httpEntity.getContent());
            this.httpEntity = httpEntity;
        }

        @Override
        public void close() throws IOException {
            try {
                super.close();
            } finally {
                try {
                    EntityUtils.consume(httpEntity);
                } catch (Throwable exc) {
                    ExceptionUtils.throwIfUnrecoverable(exc);
                    logger.error("Unable to consume httpEntity: " + exc, exc);
                }
            }
        }
    }
}
