package ru.yandex.chemodan.util.http;

import javax.servlet.http.Cookie;

import org.junit.Test;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.oauth.OauthClient;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.web.servlet.mock.MockHttpServletRequest;

/**
 * @author buberman
 */
public class YandexUidCookieParserTest {
    private static final String YUID_VALUE_VALID = "123456789012345678";
    private static final String YUID_VALUE_TOO_LONG = "123456789012345678901234567890";
    private static final String YUID_VALUE_TOO_LONG_EXPECTED =
            YUID_VALUE_TOO_LONG.substring(0, YandexUidCookieParser.MAX_YUID_LENGTH - 3) + "...";

    /**
     * Test YUID parsing when it's passed as a cookie: too long must be trimmed, valid must remain unchanged.
     */
    @Test
    public void testParseYandexUidCodeFromCookie() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setCookies(new Cookie(YandexUidCookieParser.YANDEXUID_COOKIE_NAME, YUID_VALUE_VALID));

        HttpServletRequestX requestX = new HttpServletRequestX(request);

        Option<String> yuid = YandexUidCookieParser.parseYandexUidCode(requestX);

        Assert.isTrue(yuid.isPresent(), "YUID must be present.");
        Assert.equals(YUID_VALUE_VALID, yuid.get(), "Valid YUID must be passed unaltered");

        request.setCookies(new Cookie(YandexUidCookieParser.YANDEXUID_COOKIE_NAME, YUID_VALUE_TOO_LONG));

        yuid = YandexUidCookieParser.parseYandexUidCode(requestX);

        Assert.isTrue(yuid.isPresent(), "YUID must be present, but trimmed.");
        Assert.equals(YUID_VALUE_TOO_LONG_EXPECTED, yuid.get(), "Too long YUID must be trimmed to length");

    }

    /**
     * Test YUID parsing when it's cookie is passed as a request parameter:
     * too long must be trimmed, valid must remain unchanged.
     */
    @Test
    public void testParseYandexUidCodeFromRequestParam() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        request.setCookies();
        request.addHeader(OauthClient.YA_CLIENT_COOKIE_HEADER, YandexUidCookieParser.YANDEXUID_COOKIE_NAME + "="
                + YUID_VALUE_VALID);

        HttpServletRequestX requestX = new HttpServletRequestX(request);

        Option<String> yuid = YandexUidCookieParser.parseYandexUidCode(requestX);

        Assert.isTrue(yuid.isPresent(), "YUID must be present.");
        Assert.equals(YUID_VALUE_VALID, yuid.get(), "Valid YUID must be passed unaltered");


        request = new MockHttpServletRequest();
        request.setCookies();
        request.addHeader(OauthClient.YA_CLIENT_COOKIE_HEADER, YandexUidCookieParser.YANDEXUID_COOKIE_NAME + "="
                + YUID_VALUE_TOO_LONG);
        requestX = new HttpServletRequestX(request);

        yuid = YandexUidCookieParser.parseYandexUidCode(requestX);

        Assert.isTrue(yuid.isPresent(), "YUID must be present, but trimmed.");
        Assert.equals(YUID_VALUE_TOO_LONG_EXPECTED, yuid.get(), "Too long YUID must be trimmed to length");

    }
}
