package ru.yandex.chemodan.util.http.zora;

import java.io.IOException;

import lombok.AllArgsConstructor;
import org.apache.http.Header;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.HttpResponseInterceptor;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.HttpContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.tvm2.SingleClientResolver;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */

public class ZoraHttpClientConfigurator extends HttpClientConfigurator {
    @Autowired(required = false)
    private ZoraConfiguration customConfiguration;
    @Autowired
    private Tvm2 tvm2;

    @Value("${tvm.zora.client-id}")
    private int zoraClientId;
    @Value("${zora.host}")
    private String zoraHost;
    @Value("${zora.port}")
    private int zoraPort;


    @Override
    public ApacheHttpClientUtils.Builder createBuilder() {
        ListF<ZoraHeader> defaultHeaders = Cf.list(
                new ZoraHeader.ForceGzip(),
                new ZoraHeader.IgnorCerts(true),
                new ZoraHeader.Redirs(true)
        );

        ListF<ZoraHeader> customHeaders = Option.ofNullable(customConfiguration)
                .map(ZoraConfiguration::getHeaders).getOrElse(Cf.list());

        ZoraRoutePlanner routePlanner = new ZoraRoutePlanner(zoraHost, zoraPort);

        ListF<ZoraHeader> headersWithOverrides = defaultHeaders.toMap(ZoraHeader::getName, h -> h)
                .plus(customHeaders.toMap(ZoraHeader::getName, h -> h)).values().toList();


        Function2V<HttpClientBuilder, RequestConfig.Builder> customConfig =
                (clientBuilder, requestBuilder) -> clientBuilder
                        .setRoutePlanner(routePlanner)
                        .addInterceptorLast(new ZoraErrorsInterceptor())
                        .addInterceptorLast(new ZoraHeadersInterceptor(headersWithOverrides))
                        .addInterceptorLast(tvm2.serviceTicketInterceptor(new SingleClientResolver(zoraClientId)));

        return super.createBuilder().withCustomConfig(customConfig);
    }

    public static class ZoraErrorsInterceptor implements HttpResponseInterceptor {
        public static final Logger logger = LoggerFactory.getLogger(ZoraErrorsInterceptor.class);

        @Override
        public void process(HttpResponse response, HttpContext context) throws IOException {
            if (response.getEntity() == null) {
                logger.warn("Received response with no entity from zora");
            }
            if (response.getStatusLine().getStatusCode() == 599) {
                findZoraErrorHeaders(response.getAllHeaders());
            }
        }

        private static void findZoraErrorHeaders(Header[] headers) {
            String errorCode = null;
            String errorDescription = "";

            for (Header header : headers) {
                if (header.getName().equalsIgnoreCase("X-Ya-GoZora-Error-Code")) {
                    errorCode = header.getValue();
                } else if (header.getName().equalsIgnoreCase("X-Yandex-Gozora-Error-Description")) {
                    errorDescription = header.getValue();
                }
            }

            if (errorCode != null) {
                throw new ZoraException(errorCode, errorDescription);
            }
        }
    }

    @AllArgsConstructor
    private static class ZoraHeadersInterceptor implements HttpRequestInterceptor {
        private final ListF<ZoraHeader> headers;

        @Override
        public void process(HttpRequest request, HttpContext context) {
            headers.forEach(request::setHeader);
        }
    }
}
