package ru.yandex.chemodan.util.idm;

import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.bender.serialize.BenderSerializer;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
public class IdmTests {
    private BenderSerializer<RoleNode> serializer = IdmUtils.mapper.createSerializer(RoleNode.class);

    @Test
    public void createSimpleSlug() {
        ListF<RoleDescription> roles = Cf.list(RoleDescription.simple("moderator"), RoleDescription.simple("admin"));
        SlugNode single = SlugNode.definitive("group", MultilingualString.singleValue("group"), roles, false);
        RoleNode info = RoleNode.root(single);

        String serialization = new String(serializer.serializeJson(info));

        Assert.equals("{" +
                "\"code\":0," +
                "\"roles\":{" +
                    "\"slug\":\"group\"," +
                    "\"name\":\"group\"," +
                    "\"values\":{" +
                        "\"moderator\":{\"name\":\"moderator\"}," +
                        "\"admin\":{\"name\":\"admin\"}" +
                    "}" +
                "}}", serialization);
    }

    @Test
    public void createSimpleHierarchicSlug() {
        ListF<RoleDescription> subsRoles =
                Cf.list(RoleDescription.simple("developer"),
                        RoleDescription.simple("manager"));
        SlugNode subsSlug = SlugNode.definitiveDefault(subsRoles);
        RoleNode subs = RoleNode.intermediate(MultilingualString.singleValue("subscriptions"), subsSlug);

        ListF<RoleDescription> rulesRoles =
                Cf.list(RoleDescription.simple("admin"),
                        RoleDescription.simple("auditor"));
        SlugNode rulesSlug = SlugNode.definitiveDefault(rulesRoles);
        RoleNode rules = RoleNode.intermediate(MultilingualString.singleValue("IDM"), rulesSlug);

        ListF<RoleDescription> secretRoles =
                Cf.list(RoleDescription.simple("kgb"),
                        RoleDescription.simple("cia"),
                        RoleDescription.simple("mi6"));
        SlugNode secretSlug = SlugNode.definitiveDefault(secretRoles);
        RoleNode secrets = RoleNode.intermediate(MultilingualString.singleValue("Infosec"), secretSlug).hidden();

        SlugNode project = new SlugNode("project", MultilingualString.singleValue("project"),
                Cf.map("subs", subs, "rules", rules, "secret_group", secrets));

        RoleNode info = RoleNode.root(project);

        String serialization = new String(serializer.serializeJson(info));

        String expected ="{" +
                "\"code\":0," +
                "\"roles\":{" +
                    "\"slug\":\"project\"," +
                    "\"name\":\"project\"," +
                    "\"values\":{" +
                        "\"subs\":{" +
                            "\"name\":\"subscriptions\"," +
                            "\"roles\":{" +
                                "\"slug\":\"role\"," +
                                "\"name\":{\"en\":\"role\",\"ru\":\"роль\"}," +
                                "\"values\":{" +
                                    "\"manager\":{\"name\":\"manager\"}," +
                                    "\"developer\":{\"name\":\"developer\"}" +
                                "}" +
                            "}" +
                        "}," +
                        "\"rules\":{" +
                            "\"name\":\"IDM\"," +
                            "\"roles\":{" +
                                "\"slug\":\"role\"," +
                                "\"name\":{\"en\":\"role\",\"ru\":\"роль\"}," +
                                "\"values\":{" +
                                    "\"auditor\":{\"name\":\"auditor\"}," +
                                    "\"admin\":{\"name\":\"admin\"}" +
                                "}" +
                            "}" +
                        "}," +
                        "\"secret_group\":{" +
                            "\"name\":\"Infosec\"," +
                            "\"visibility\":false," +
                            "\"roles\":{" +
                                "\"slug\":\"role\"," +
                                "\"name\":{\"en\":\"role\",\"ru\":\"роль\"}," +
                                "\"values\":{" +
                                    "\"kgb\":{\"name\":\"kgb\"}," +
                                    "\"cia\":{\"name\":\"cia\"}," +
                                    "\"mi6\":{\"name\":\"mi6\"}" +
                                "}" +
                            "}" +
                        "}" +
                    "}" +
                "}" +
                "}";

        Assert.equals(expected, serialization);
    }

    @Test
    public void parseRoleSpecification() {
        String serializedSpec = "{\"project\": \"subs\", \"role\": \"manager\"}";
        RoleSpecification spec = new RoleSpecificationConverter().convert(serializedSpec, null);
        RoleSpecification expected = new RoleSpecification(Tuple2List.fromPairs("project", "subs", "role", "manager"));
        Assert.equals(expected, spec);
    }
}
