package ru.yandex.chemodan.util.jdbc;

import org.apache.commons.lang3.Validate;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.function.Function;
import ru.yandex.commune.alive2.location.ConductorYpQloudLocationResolver;
import ru.yandex.commune.alive2.location.DcUtils;
import ru.yandex.commune.alive2.location.Location;
import ru.yandex.commune.alive2.location.LocationResolver;
import ru.yandex.commune.db.shard.ShardInfo;
import ru.yandex.inside.admin.conductor.Conductor;
import ru.yandex.inside.admin.conductor.GroupOrHost;
import ru.yandex.inside.admin.conductor.NotFoundConductorException;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.regex.Matcher2;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.time.DurationOrInfinity;

/**
 * @author tolmalev
 */
public class ConductorDbListUtils {
    private static final Logger logger = LoggerFactory.getLogger(ConductorDbListUtils.class);

    private static final Pattern2 hostPattern =
            Pattern2.compile("^[^\\d]*(\\d+)([^\\.]*)\\.(cmail|mail|disk)\\.yandex\\.net$");
    private static final Pattern2 dcPrefixPattern = Pattern2.compile("^([a-z]+)");

    public static ListF<ShardInfo> getConfiguration(LocationResolver locationResolver,
            GroupOrHost groupOrHost, IpPort port, String dbName)
    {
        if (groupOrHost.isGroup()) {
            return parse(locationResolver.getGroupHosts(groupOrHost.getName()), port, dbName);
        } else {
            return singleHost(groupOrHost.getName(), port, dbName);
        }
    }

    public static ListF<ShardInfo> parse(ListF<String> hosts, IpPort port, String dbName) {
        MapF<Integer, ListF<HostInfo>> dataByShardId = hosts.map(parseF())
                .groupBy(HostInfo.shardIdF());

        ListF<ShardInfo> list = Cf.arrayList();

        for (int shardId: dataByShardId.keys()) {
            ShardInfo shard = new ShardInfo();

            shard.setId(shardId);
            shard.setName("data");

            ListF<String> readers = Cf.arrayList();
            for (HostInfo host : dataByShardId.getTs(shardId)) {
                readers.add(formatDsUrl(host.host, port, dbName));
            }

            shard.setReaderHostPortDbnames(readers);
            shard.setWriterHostPortDbname(readers.first());

            list.add(shard);
        }

        return list;
    }

    public static ListF<ShardInfo> singleHost(String host, IpPort port, String dbName) {
        ShardInfo shard = new ShardInfo();

        int shardId = 1;

        shard.setId(shardId);
        shard.setName("data");
        shard.setWriterHostPortDbname(formatDsUrl(host, port, dbName));
        shard.setReaderHostPortDbnames(Cf.list());

        return Cf.list(shard);
    }

    public static String formatDsUrl(String host, IpPort port, String dbName) {
        return StringUtils.format("{}:{}/{}", host, port.getPort(), dbName);
    }

    public static boolean inSameDc(Location location, String hostname2, LocationResolver locationResolver) {
        String dcname2 = null;

        try {
            ConductorYpQloudLocationResolver conductorLocationResolver = (ConductorYpQloudLocationResolver) locationResolver;
            dcname2 = getDcOrNull(hostname2, conductorLocationResolver.getConductor());
        } catch (RuntimeException e) {
             logger.warn("DC by conductor resolving error", e);
        }

        if (dcname2 != null) {
            return location.dcName.isSome(dcname2);
        } else {
            return location.dcName.isMatch(dc -> DcUtils.getDcByShortName(dc).equals(DcUtils.getDc(hostname2)));
        }
    }

    private static String getDcOrNull(String hostname1, Conductor conductor) {
        try {
            return conductor
                    .getDcNameByHost(UrlUtils.extractHost(hostname1),
                            new DurationOrInfinity(Duration.standardDays(1)),
                            DurationOrInfinity.infty())
                    .map(parseDcPrefixF())
                    .getOrNull();
        } catch (NotFoundConductorException e) {
            return null;
        }
    }

    static HostInfo parse(String hostname) {
        Matcher2 matcher = hostPattern.matcher2(hostname);
        Validate.isTrue(matcher.matches(), "Bad hostname: " + hostname);
        return new HostInfo(
                matcher.group(0).get(),
                matcher.group(2).get(),
                Cf.Integer.parse(matcher.group(1).get())
        );
    }

    static Function<String, HostInfo> parseF() {
        return ConductorDbListUtils::parse;
    }

    static Function<String, String> parseDcPrefixF() {
        return dcname -> dcPrefixPattern.findFirstGroup(dcname).getOrThrow("Invalid DC name " + dcname);
    }

    static class HostInfo extends DefaultObject {
        public final String host;
        public final String dc;
        public final int shardId;

        HostInfo(String host, String dc, int shardId) {
            this.host = host;
            this.dc = dc;
            this.shardId = shardId;
        }

        static Function<HostInfo, Integer> shardIdF() {
            return hostInfo -> hostInfo.shardId;
        }
    }
}
