package ru.yandex.chemodan.util.jdbc;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.boot.value.OverridableValue;
import ru.yandex.chemodan.boot.value.OverridableValuePrefixAware;
import ru.yandex.misc.db.CompoundDSProperties;
import ru.yandex.misc.db.CompoundExtra;
import ru.yandex.misc.db.DSProperties;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.Validate;

/**
 * @author vpronto
 */
@Data
@Builder(toBuilder = true)
@AllArgsConstructor
@NoArgsConstructor
public class DataSourceProperties extends CompoundDSProperties implements OverridableValuePrefixAware {
    private CamelWords propertyPrefix = CamelWords.valueOf(Cf.list(""));

    @OverridableValue("${jdbc.pool.max.active}")
    private int maxActive;

    @OverridableValue("${jdbc.pool.max.idle}")
    private int maxIdle;

    @OverridableValue("${jdbc.pool.min.idle}")
    private int minIdle;

    @OverridableValue("${jdbc.pool.read.max.active}")
    private int readMaxActive;

    @OverridableValue("${jdbc.pool.read.min.idle}")
    private int readMinIdle;

    @OverridableValue("${jdbc.pool.sql-admin.max.active}")
    private int sqlAdminMaxActive;

    @OverridableValue("${jdbc.pool.sql-admin.min.idle}")
    private int sqlAdminMinIdle;

    @OverridableValue("${jdbc.pool.max.wait.millis}")
    private long maxWaitMillis;

    @OverridableValue("${jdbc.pool.ps.pooling}")
    private boolean psPooling;

    @OverridableValue("${jdbc.pool.test.on.borrow}")
    private boolean testOnBorrow;

    @OverridableValue("${jdbc.pool.test.when.idle}")
    private boolean testWhenIdle;

    @OverridableValue("${jdbc.pool.ping.max.wait.secs}")
    private int pingMaxWaitSecs;

    @OverridableValue("${jdbc.pool.time.between.eviction.runs.millis}")
    private long timeBetweenEvictionRunsMillis;

    @OverridableValue("${jdbc.pool.num.tests.per.eviction.run}")
    private int numTestsPerEvictionRun;

    @OverridableValue("${pg.url.prefix}")
    private String urlPrefix;

    @OverridableValue("${pg.url.suffix}")
    private String urlSuffix;

    @OverridableValue("${pg.ping.url.params}")
    private String pingUrlParams;

    @OverridableValue("${pg.username}")
    private String username;

    @OverridableValue("${pg.password}")
    private String password;

    @Value("${pg.read.username}")
    private String readUsername;

    @Value("${pg.read.password}")
    private String readPassword;

    @OverridableValue("${pg.sql-admin.username}")
    private String sqlAdminUsername;

    @OverridableValue("${pg.sql-admin.password}")
    private String sqlAdminPassword;

    @OverridableValue("${pgpass.path}")
    private String pgpass;

    @OverridableValue("${pg.replication.maximumLag}")
    private int replicationMaximumLag;

    @OverridableValue("${pg.delay.between.pings}")
    private long delayBetweenPingsMillis;

    @OverridableValue("${pg.hosts}")
    private String hosts;

    @OverridableValue("${pg.dbname}")
    private String dbName;

    @OverridableValue("${pg.port}")
    private IpPort port;

    @OverridableValue("${pgaas.host}")
    private String pgaasHost;

    @OverridableValue("${pgaas.port}")
    private IpPort pgaasPort;

    @OverridableValue("${jdbc.long-connection-acquisition}")
    private Duration longConnectionAcquisition;

    @Override
    public long getValidationTimeout() {
        return maxWaitMillis / 2;
    }

    @Override
    public void setOverridableValuePrefix(String prefix) {
        this.propertyPrefix = CamelWords.valueOf(
                Cf.x(prefix.split("\\."))
                        .map(s -> s.replace("-", "").replace("_", "").replace(".", ""))
        );
    }

    public String getOverridableValuePrefix() {
        return propertyPrefix.toDbName();
    }

    public void setExtraDSPropertiesIfSetUp() {
        setExtraProperties(new CompoundExtra<>(
                createDSProperties(new ExtraDsProperties(
                        readUsername, readPassword, readMaxActive, readMinIdle)),
                createDSProperties(new ExtraDsProperties(
                        sqlAdminUsername, sqlAdminPassword, sqlAdminMaxActive, sqlAdminMinIdle))
        ));
    }

    private Option<DSProperties> createDSProperties(ExtraDsProperties props) {
        if (props.username.isEmpty()) {
            return Option.empty();
        }
        Validate.notEmpty(props.password, "Password was not provided for " + props.username);

        return Option.of(toBuilder()
                .username(props.username)
                .password(props.password)
                .maxActive(props.maxActive >= 0 ? props.maxActive : maxActive)
                .minIdle(props.minIdle >= 0 ? props.minIdle : minIdle)
                .build());
    }

    @Data
    private static class ExtraDsProperties {
        private final String username;
        private final String password;
        private final int maxActive;
        private final int minIdle;
    }
}
