package ru.yandex.chemodan.util.jdbc;

import java.util.function.Supplier;

import javax.sql.DataSource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.jdbc.logging.LastAccessedDsAwareQueryInterceptor;
import ru.yandex.chemodan.util.jdbc.logging.LoggingQueryInterceptor;
import ru.yandex.chemodan.util.jdbc.logging.LoggingQueryInterceptorConfiguration;
import ru.yandex.chemodan.util.sharpei.DynamicShardMetaNotifier;
import ru.yandex.chemodan.util.sharpei.ShardInfoSource;
import ru.yandex.commune.alive2.location.LocationResolver;
import ru.yandex.commune.db.shard.ShardInfo;
import ru.yandex.commune.db.shard.spring.ShardResolver;
import ru.yandex.commune.db.shard2.ShardManager2;
import ru.yandex.commune.db.shard2.ShardMetaNotifier;
import ru.yandex.commune.db.shard2.ShardMetaStaticNotifierSupport;
import ru.yandex.inside.admin.conductor.GroupOrHost;
import ru.yandex.misc.db.PooledDSFactorySupportConfigurator;
import ru.yandex.misc.db.pool.PooledDataSourceFactorySupport;
import ru.yandex.misc.db.postgres.PgBouncerFamiliarTransactionManager;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.spring.jdbc.ArgPreparedStatementSetter;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author tolmalev
 */
public class JdbcDatabaseConfigurator {

    private final LocationResolver locationResolver;

    private final LoggingQueryInterceptorConfiguration loggingQueryConf;

    protected final DataSourceProperties dataSourceProperties;

    //TODO: use spring instead
    private final PooledDSFactorySupportConfigurator configureJdbcFactory;
    private Option<ListF<String>> configuredHostPortDbNames = Option.empty();

    public JdbcDatabaseConfigurator(
            LocationResolver locationResolver,
            LoggingQueryInterceptorConfiguration loggingQueryConf,
            DataSourceProperties dataSourceProperties)
    {
        overrideCredentials(dataSourceProperties);

        this.locationResolver = locationResolver;
        this.loggingQueryConf = loggingQueryConf;
        this.dataSourceProperties = dataSourceProperties;
        this.configureJdbcFactory = new CompositePooledDSFactorySupportConfigurator(locationResolver);
    }

    public DataSourceProperties getDataSourceProperties() {
        return dataSourceProperties;
    }

    private void overrideCredentials(DataSourceProperties dataSourceProperties) {
        PgCredentials credentials = PgCredentials.resolve(dataSourceProperties.getUsername(),
                dataSourceProperties.getPassword(),
                dataSourceProperties.getPgpass(),
                dataSourceProperties.getDbName());

        PgCredentials.User mainUser = credentials.getMainUser();

        dataSourceProperties.setUsername(mainUser.getUsername());
        dataSourceProperties.setPassword(mainUser.getPassword());

        credentials.getReadOnlyUser().forEach(user -> {
            dataSourceProperties.setReadUsername(user.getUsername());
            dataSourceProperties.setReadPassword(user.getPassword());
        });

        dataSourceProperties.setExtraDSPropertiesIfSetUp();
    }

    public ShardManager2 configureSharded(ShardResolver shardResolver) {
        return configureSharded(shardResolver, consTemplateParams());
    }

    public ShardManager2 configureSharded(ShardResolver shardResolver, JdbcTemplateParams templateParams) {
        return configureSharded(shardResolver, templateParams, configureShardMetaNotifier());
    }

    public ShardManager2 configureSharded(ShardResolver shardResolver, JdbcTemplateParams templateParams,
            Supplier<LastAccessedDsAwareQueryInterceptor> lastAccessedDsAwareQueryInterceptorF)
    {
        return configureSharded(shardResolver, templateParams, configureShardMetaNotifier(),
                lastAccessedDsAwareQueryInterceptorF);
    }

    private ShardManager2 configureSharded(ShardResolver shardResolver, JdbcTemplateParams templateParams,
            ShardMetaNotifier shardMetaNotifier)
    {
        return configureSharded(shardResolver, templateParams, shardMetaNotifier,
                () -> new LoggingQueryInterceptor(loggingQueryConf));
    }

    public ListF<Integer> getShardIds() {
        return getShards()
                .map(ShardInfo::getId);
    }

    private ListF<ShardInfo> getShards() {
        ListF<ShardInfo> shards = GroupOrHost.parseListFromString(dataSourceProperties.getHosts())
                .flatMap(groupOrHost -> ConductorDbListUtils.getConfiguration(
                        locationResolver, groupOrHost,
                        dataSourceProperties.getPort(),
                        dataSourceProperties.getDbName()));

        shards.forEach(s -> s.setName(dataSourceProperties.getPropertyPrefix().toXmlName()));

        return shards;
    }

    private ShardManager2 configureSharded(
            ShardResolver shardResolver,
            JdbcTemplateParams templateParams,
            ShardMetaNotifier shardMetaNotifier,
            Supplier<LastAccessedDsAwareQueryInterceptor> lastAccessedDsAwareQueryInterceptorF)
    {
        return new ShardManager2(
                shardResolver,
                shardMetaNotifier,
                shard -> configureDataSource(shard, lastAccessedDsAwareQueryInterceptorF),
                templateParams::consTemplate,
                PgBouncerFamiliarTransactionManager::new,
                Option.empty()
        );
    }

    public JdbcTemplate3 consTemplate(DataSource dataSource) {
        return consTemplateParams().consTemplate(dataSource);
    }

    public JdbcTemplateParams consTemplateParams() {
        return new JdbcTemplateParams(
                ru.yandex.chemodan.util.jdbc.logging.QueryInterceptors::defaultQueryInterceptor,
                ArgPreparedStatementSetter::new);
    }

    public DataSource configureDataSource() {
        Validate.notBlank(dataSourceProperties.getHosts());
        return configureDataSource(
                locationResolver.resolveHostsFromString(dataSourceProperties.getHosts())
                        .map(host -> ConductorDbListUtils.formatDsUrl(host,
                                dataSourceProperties.getPort(),
                                dataSourceProperties.getDbName()))
        );
    }

    public Option<ListF<String>> getConfiguredHostPortDbNames() {
        return configuredHostPortDbNames;
    }

    private DataSource configureDataSource(ListF<String> hostPortDbNames) {
        return configureDataSource(hostPortDbNames, () -> new LoggingQueryInterceptor(loggingQueryConf));
    }

    protected DataSource configureDataSource(
            ShardInfo shard, Supplier<LastAccessedDsAwareQueryInterceptor> loggingQueryInterceptorF)
    {
        return configureDataSource(
                Cf.list(shard.getWriterHostPortDbname()).plus(shard.getReaderHostPortDbnames()),
                loggingQueryInterceptorF);
    }

    private DataSource configureDataSource(
            ListF<String> hostPortDbNames, Supplier<LastAccessedDsAwareQueryInterceptor> loggingQueryInterceptorF)
    {
        configuredHostPortDbNames = Option.of(hostPortDbNames);

        PooledDataSourceFactorySupport dbcpFactory = configureJdbcFactory.configure(dataSourceProperties);

        DcAwareDynamicMasterSlaveDataSourceFactory factory = new DcAwareDynamicMasterSlaveDataSourceFactory(
                locationResolver, dataSourceProperties.getReplicationMaximumLag(),
                dataSourceProperties.getDelayBetweenPingsMillis(),
                loggingQueryInterceptorF.get());

        factory.setDataSourceFactory(dbcpFactory);
        factory.setDsUrlPrefix(dataSourceProperties.getUrlPrefix());
        factory.setDsUrlSuffix(dataSourceProperties.getUrlSuffix());
        factory.setUsername(dataSourceProperties.getUsername());
        factory.setPassword(dataSourceProperties.getPassword());
        factory.setQuietMode(true);
        factory.setHostPortDbNames(hostPortDbNames);

        factory.afterPropertiesSet();

        return factory.getObject();
    }

    protected ShardMetaNotifier configureShardMetaNotifier() {
        ListF<ShardInfo> shards = getShards();
        return new ShardMetaStaticNotifierSupport() {
            @Override
            protected ListF<ShardInfo> getShards() {
                return shards;
            }

            @Override
            public String getShardTypeName() {
                return dataSourceProperties.getPropertyPrefix().toXmlName();
            }
        };
    }

    public static class DynamicConfigurator extends JdbcDatabaseConfigurator {

        private final ShardInfoSource shardInfoSource;

        public DynamicConfigurator(LocationResolver locationResolver,
                                   LoggingQueryInterceptorConfiguration loggingQueryConf,
                                   DataSourceProperties dataSourceProperties,
                                   ShardInfoSource shardInfoSource)
        {
            super(locationResolver, loggingQueryConf, dataSourceProperties);
            this.shardInfoSource = shardInfoSource;
        }

        @Override
        protected DynamicShardMetaNotifier configureShardMetaNotifier() {
            DynamicShardMetaNotifier shardMetaNotifier = new DynamicShardMetaNotifier(shardInfoSource);
            // XXX: bad place to start DWSBS but it will work
            shardMetaNotifier.start();
            return shardMetaNotifier;
        }

        @Override
        public ListF<Integer> getShardIds() {
            return getShardInfoSource()
                    .getShards()
                    .map(ShardInfo::getId);
        }

        @Override
        protected DataSource configureDataSource(
                ShardInfo shard, Supplier<LastAccessedDsAwareQueryInterceptor> loggingQueryInterceptorF)
        {
            return new EqualByShardDcAwareDynamicMasterSlaveDataSource(
                    dataSourceProperties.getPropertyPrefix().toXmlName() + "." + shard.getId(),
                    (DcAwareDynamicMasterSlaveDataSource) super.configureDataSource(shard, loggingQueryInterceptorF));
        }

        public ShardInfoSource getShardInfoSource() {
            return shardInfoSource;
        }
    }
}
