package ru.yandex.chemodan.util.jdbc;

import javax.sql.DataSource;

import com.zaxxer.hikari.HikariDataSource;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.util.jdbc.logging.LoggingQueryInterceptorConfiguration;
import ru.yandex.commune.alive2.location.LocationResolver;
import ru.yandex.misc.db.hikari.HikariPooledDS;
import ru.yandex.misc.db.masterSlave.MasterSlaveDataSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
public class JdbcDatabaseConfiguratorTest {
    @Test
    public void configureReadWriteDataSource() {
        ListF<PgCredentials.User> users =
                Cf.list(new PgCredentials.User("user", "pass"), new PgCredentials.User("user_read", "pass2"));
        assertDatasource(users);
    }

    @Test
    public void configureSimpleDataSource() {
        ListF<PgCredentials.User> users = Cf.list(new PgCredentials.User("user", "pass"));
        assertDatasource(users);
    }

    private void assertDatasource(ListF<PgCredentials.User> pgpassUsers) {
        DataSourceProperties properties = new DataSourceProperties();
        properties.setHosts("localhost");
        properties.setPgaasPort(IpPort.cons(6432));
        properties.setPort(IpPort.cons(6432));
        properties.setDbName("dbmane");
        properties.setMaxActive(1);
        properties.setMaxWaitMillis(1000);
        properties.setUsername("default_username");
        properties.setPassword("default_password");
        properties.setUrlPrefix("jdbc:postgresql://");
        properties.setReadUsername("");
        properties.setReadPassword("");
        properties.setReadMaxActive(1);
        properties.setSqlAdminUsername("");
        properties.setSqlAdminPassword("");

        LocationResolver locationResolver = Mockito.mock(LocationResolver.class);

        Mockito.when(locationResolver.resolveHostsFromString(Mockito.any())).thenReturn(Cf.list("localhost"));

        LoggingQueryInterceptorConfiguration loggingQueryConf =
                Mockito.mock(LoggingQueryInterceptorConfiguration.class);

        DataSource dataSource = File2.withNewTempFile(file -> {
            PgCredentialsTest.fillPgpassFile(file, properties.getDbName(), pgpassUsers);
            properties.setPgpass(file.getAbsolutePath());
            JdbcDatabaseConfigurator configurator =
                    new JdbcDatabaseConfigurator(locationResolver, loggingQueryConf, properties);
             return configurator.configureDataSource();
        });

        ListF<DataSource> allDataSources = Cf.list();
        if (dataSource instanceof DcAwareDynamicMasterSlaveDataSource) {
            allDataSources = ((DcAwareDynamicMasterSlaveDataSource) dataSource).getAllDataSources();
        } else if (dataSource instanceof MasterSlaveDataSource) {
            allDataSources = ((MasterSlaveDataSource) dataSource).getAllDataSources();
        }

        ListF<HikariPooledDS> hikariDataSources = allDataSources.filterByType(HikariPooledDS.class);

        Assert.notEmpty(hikariDataSources);

        PgCredentials.User mainDSUserExpected = pgpassUsers.filterNot(PgCredentials.User::isRead).first();
        Option<PgCredentials.User> readDSUserExpected = pgpassUsers.filter(PgCredentials.User::isRead).firstO();

        boolean expectedHasReadDSStatus = readDSUserExpected.isPresent();

        hikariDataSources.forEach(hikariDataSource -> {
            PgCredentials.User mainDSUser = extractUserFromDS(hikariDataSource.getMainHikariDataSourceForTest());
            Option<PgCredentials.User> readDSUser =
                    hikariDataSource.getReadHikariDataSourceForTest().map(this::extractUserFromDS);

            Assert.equals(mainDSUserExpected, mainDSUser);
            Assert.equals(readDSUserExpected, readDSUser);
        });

        SetF<Boolean> hasReadDataSourceStatuses = hikariDataSources.map(HikariPooledDS::hasReadDataSource).unique();
        Assert.hasSize(1, hasReadDataSourceStatuses, "All datasources must have same hasReadDataSource status");
        Assert.equals(expectedHasReadDSStatus, hasReadDataSourceStatuses.single());
    }

    private PgCredentials.User extractUserFromDS(HikariDataSource dataSource) {
        return new PgCredentials.User(dataSource.getUsername(), dataSource.getPassword());
    }
}
