package ru.yandex.chemodan.util.jdbc;

import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
public class PgCredentialsTest {

    private final static String DBNAME = "dbname";
    private final static PgCredentials.User DEFAULT_USER = new PgCredentials.User("username", "password");

    @Test
    public void withoutUsers() {
        withTempPgpassFile(Cf.list(), credentials -> {
            Assert.none(credentials.getReadOnlyUser());
            Assert.equals(DEFAULT_USER, credentials.getMainUser());
        });
    }

    @Test
    public void onlyMainUser() {
        PgCredentials.User user = new PgCredentials.User("custom_user", "custom_pass");
        withTempPgpassFile(Cf.list(user), credentials -> {
            Assert.none(credentials.getReadOnlyUser());
            Assert.equals(user, credentials.getMainUser());
        });
    }

    @Test
    public void onlyReadOnlyUser() {
        PgCredentials.User readOnlyUser = new PgCredentials.User("custom_user_read", "custom_pass");
        withTempPgpassFile(Cf.list(readOnlyUser), credentials -> {
            Assert.some(readOnlyUser, credentials.getReadOnlyUser());
            Assert.equals(DEFAULT_USER, credentials.getMainUser());
        });
    }

    @Test
    public void bothUsers() {
        PgCredentials.User writeUser = new PgCredentials.User("custom_user", "custom_pass");
        PgCredentials.User readOnlyUser = new PgCredentials.User("custom_user_read", "another_custom_pass");
        withTempPgpassFile(Cf.list(readOnlyUser, writeUser), credentials -> {
            Assert.some(readOnlyUser, credentials.getReadOnlyUser());
            Assert.equals(writeUser, credentials.getMainUser());
        });
    }

    protected static void fillPgpassFile(File2 file, String dbname, ListF<PgCredentials.User> users) {
        String content = users
                .map(user -> Cf.list("*", "6432", dbname, user.getUsername(), user.getPassword()).mkString(":"))
                .mkString("\n");
        file.write(content);
    }

    private void withTempPgpassFile(ListF<PgCredentials.User> users, Function1V<PgCredentials> action) {
        File2.withNewTempFile(file -> {
            fillPgpassFile(file, DBNAME, users);
            PgCredentials credentials = PgCredentials.resolve(
                    DEFAULT_USER.getUsername(), DEFAULT_USER.getPassword(), file.getAbsolutePath(), DBNAME);
            action.apply(credentials);
        });
    }
}
