package ru.yandex.chemodan.util.jdbc.logging;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.spring.jdbc.intercept.QueryInfo;
import ru.yandex.misc.spring.jdbc.intercept.QueryResultInfo;
import ru.yandex.misc.time.MoscowTime;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author tolmalev
 */
public class LoggingQueryInterceptor implements LastAccessedDsAwareQueryInterceptor {

    private final LoggingQueryInterceptorConfiguration configuration;
    private final LastAccessedDsHolder lastDs;

    public LoggingQueryInterceptor(LoggingQueryInterceptorConfiguration configuration) {
        this.configuration = configuration;
        this.lastDs = new LastAccessedDsHolder();
    }

    @Override
    public Object queryStarted(QueryInfo q) {
        if (q.getArgs().isPresent()) {
            ListF<Object> args = q.getArgs().get();
            for (int i = 0; i < args.length(); ++i) {
                Object ith = args.get(i);
                if (ith instanceof Instant) {
                    args.set(i, ((Instant) ith).toDateTime(MoscowTime.TZ));
                }
            }
        }
        if (configuration.logger.isTraceEnabled()) {
            configuration.logger.trace("Q: " + q.getQFormatted());
        }
        return System.currentTimeMillis();
    }

    @Override
    public void queryCompleted(Object tag, QueryResultInfo result) {
        long elapsed = System.currentTimeMillis() - (Long) tag;
        QueryInfo queryInfo = result.getQueryInfo();

        Level level = queryInfo.isSelect() ? Level.DEBUG : Level.INFO; // hot topic
        if (elapsed > configuration.longThreshold.get() || !result.isSuccessfully()) {
            level = Level.WARN;
        }
        if (configuration.logger.isEnabledFor(level)) {
            String message = result.getRowCount().isPresent()
                    ? "Q: {}: {} at {}, rc={}; took {}"
                    : "Q: {}: {} at {}; took {}";

            if (elapsed > configuration.longThreshold.get()) {
                message = "(LONG) " + message;
            }
            ListF<Object> args = Cf.arrayListWithCapacity(5);

            args.add(queryInfo.getQFormatted());
            args.add(result.isSuccessfully() ? "completed" : "failed");

            args.add(lastDs.get().map(info -> ""
                    + (info.isMaster ? "master " : "slave ")
                    + info.url.getHost().getOrElse("unknown")
                    + info.url.getDbName().map(n -> "/" + n).getOrElse("")).getOrElse("unknown"));

            args.addAll(result.getRowCount());
            args.add(TimeUtils.millisecondsToSecondsString(elapsed));

            configuration.logger.log(level, message, args.toArray(Object.class));
        }
    }

    public LoggingQueryInterceptorConfiguration getConfiguration() {
        return configuration;
    }

    @Override
    public LastAccessedDsHolder getLastAccessedDsHolder() {
        return lastDs;
    }
}

