package ru.yandex.chemodan.util.oauth;

import java.net.URI;
import java.util.concurrent.TimeUnit;

import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.passport.PassportApp;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadStringResponseHandler;

/**
 * @author akirakozov
 */
public class OauthClient {
    public static final String YA_CONSUMER_CLIENT_IP_HEADER = "Ya-Consumer-Client-Ip";
    public static final String YA_CLIENT_COOKIE_HEADER = "Ya-Client-Cookie";
    public static final String YA_CLIENT_HOST = "Ya-Client-Host";

    private static final OauthResponseParser RESPONSE_PARSER = new OauthResponseParser();

    private final URI baseUrl;
    private final String clientId;
    private final String clientSecret;

    private final HttpClient httpClient;

    private final String consumer;

    public enum CodeStrength {
        MEDIUM,
        LONG;

        public String getValue() {
            return this.name().toLowerCase();
        }
    }

    public OauthClient(String host, String consumer, String clientId, String clientSecret, HttpClient httpClient) {
        this(new UriBuilder(String.format("http://%s/", host)).build(), consumer, clientId, clientSecret, httpClient);
    }

    public OauthClient(PassportApp app) {
        this(app.getOauthEnvironment().url, "disk", app.id, app.secret,
                ApacheHttpClientUtils.singleConnectionClient(Option.of(Timeout.timeout(3, TimeUnit.SECONDS)), Option.empty(), true));
    }

    private OauthClient(URI baseUrl, String consumer, String clientId, String clientSecret, HttpClient httpClient) {
        this.baseUrl = baseUrl;
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.httpClient = httpClient;
        this.consumer = consumer;
    }

    public String getAuthorizationCode(
            String clientIp, String cookie, CodeStrength codeStrength, String clientHost,
            Option<Duration> ttl, Option<PassportUid> uid)
    {
        MapF<String, Object> params = getBaseParams()
                .plus1("code_strength", codeStrength.getValue())
                .plus1("consumer", consumer)
                .plus1("require_activation", false);
        if (ttl.isPresent()) {
            params = params.plus1("ttl", ttl.get().getStandardSeconds());
        }
        if (uid.isPresent()) {
            params = params.plus1("uid", uid.get().getUid());
        }

        HttpPost post = ApacheHttpClientUtils.prepareHttpPost(getUrl("/api/1/authorization_code/issue"), params);
        post.setHeader(YA_CONSUMER_CLIENT_IP_HEADER, clientIp);
        post.setHeader(YA_CLIENT_COOKIE_HEADER, cookie);
        post.setHeader(YA_CLIENT_HOST, clientHost);

        String response = ApacheHttpClientUtils.execute(post, httpClient, new ReadStringResponseHandler());
        return RESPONSE_PARSER.parseAuthorizationCodeResponse(response);
    }

    public OauthAccessToken getAccessTokenByPassword(String username, String password) {
        return getAccessToken("password", username, password);
    }

    private OauthAccessToken getAccessToken(String grantType, String username, String password) {
        MapF<String, Object> params = getBaseParams()
                .plus1("grant_type", grantType)
                .plus1("username", username)
                .plus1("password", password);

        HttpPost post = ApacheHttpClientUtils.prepareHttpPost(getUrl("/token"), params);
        String response = ApacheHttpClientUtils.execute(post, httpClient, new ReadStringResponseHandler());
        return OauthAccessToken.parseJson(response);
    }

    private URI getUrl(String path) {
        return new UriBuilder(baseUrl)
                .appendPath(path)
                .build();
    }

    private MapF<String, Object> getBaseParams() {
        return Cf.<String, Object>map()
                .plus1("client_id", clientId)
                .plus1("client_secret", clientSecret);
    }
}
