package ru.yandex.chemodan.util.ping;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.PingConfigurableServlet;
import ru.yandex.misc.worker.AlarmThread;
import ru.yandex.misc.worker.WorkerThread;

/**
 * @author tolmalev
 */
public class CoolPingServlet extends PingConfigurableServlet {
    private static final Logger logger = LoggerFactory.getLogger(CoolPingServlet.class);

    private final ListF<PingerChecker> checkers;
    private final WorkerThread checkerThread;

    private CheckResult checkResult = new CheckResult(false, "not checked yet");

    public CoolPingServlet(ListF<PingerChecker> checkers) {
        this.checkers = checkers;

        checkerThread = new AlarmThread("cool-ping-checker", 1000, false) {
            @Override
            protected void alarm() {
                updateState();
            }
        };
        checkerThread.start();
    }

    private void updateState() {
        if (!super.isAvailable()) {
            saveBadCheckResult("PingConfigurableServlet.isAvailable() -> false");
            return;
        }

        if (!super.checkCustomCondition()) {
            saveBadCheckResult("PingConfigurableServlet.checkCustomCondition() -> false. Space for logs: " + getAvailableLogSpacePercent());
            return;
        }

        ListF<PingerChecker> badCheckers = Cf.arrayList();
        for (PingerChecker checker : checkers) {
            if (!checker.isActive()) {
                badCheckers.add(checker);
            }
        }

        if (!badCheckers.isEmpty()) {
            String message = badCheckers
                    .map(checker -> "Ping checker '" + checker.toString() + "' -> false")
                    .mkString("; ");

            saveBadCheckResult(message);
        } else {
            checkResult = new CheckResult(true, "OK");
        }
    }

    private void saveBadCheckResult(String message) {
        checkResult = new CheckResult(false, message);
        logger.warn("Checker is closed: {}", message);
    }

    @Override
    public boolean isAvailable() {
        return checkResult.available;
    }

    @Override
    protected boolean checkCustomCondition() {
        CheckResult lastCheckResult = this.checkResult;
        if (!lastCheckResult.available) {
            logger.warn("/ping will respond with false. reason={}", lastCheckResult.message);
        }
        return lastCheckResult.available;
    }

    @Override
    public void stop() {
        super.stop();
        checkerThread.stopGracefully();
    }

    private static class CheckResult {
        private final boolean available;
        private final String message;

        private CheckResult(boolean available, String message) {
            this.available = available;
            this.message = message;
        }
    }
}
