package ru.yandex.chemodan.util.retry;

import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.Listeners;
import net.jodah.failsafe.RetryPolicy;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.misc.ExceptionUtils;

/**
 * @author metal
 */
public class RetryManager<T> {
    private RetryPolicy retryPolicy;
    private Listeners<T> listeners;

    public RetryManager() {
        this.retryPolicy = new RetryPolicy();
        this.listeners = new Listeners<>();
    }

    public RetryManager<T> withRetryPolicy(RetryPolicy retryPolicy) {
        this.retryPolicy = retryPolicy;
        return this;
    }

    public RetryManager<T> withRetryPolicy(int maxRetries, long delayInMillis) {
        RetryPolicy retryPolicy = new RetryPolicy()
                .withMaxRetries(maxRetries)
                .withDelay(delayInMillis, TimeUnit.MILLISECONDS);
        return withRetryPolicy(retryPolicy);
    }

    public RetryManager<T> withLogging(String message) {
        listeners.onFailedAttempt(new LoggingFailedAttemptListener<>(retryPolicy.getMaxRetries(), message));
        return this;
    }

    public RetryManager<T> withFailureCallback(Function0V errorCallback) {
        listeners.onFailure((result, failure) -> errorCallback.apply());
        return this;
    }

    public void run(Function0V function) {
        Failsafe.with(retryPolicy).with(listeners).run(function::apply);
    }

    public void runSafe(Function0V function) {
        try {
            Failsafe.with(retryPolicy).with(listeners).run(function::apply);
        } catch (Exception ignored) {
            ExceptionUtils.throwIfUnrecoverable(ignored);
        }
    }

    public T get(Function0<T> function) {
        return Failsafe.with(retryPolicy).with(listeners).get(function::apply);
    }

    public Option<T> getSafe(Function0<T> function) {
        try {
            return Option.of(Failsafe.with(retryPolicy).with(listeners).get(function::apply));
        } catch (Exception ignored) {
            ExceptionUtils.throwIfUnrecoverable(ignored);

            return Option.empty();
        }
    }

    public T get(boolean safe, Function0<T> function, Function0<T> orElseF) {
        return !safe ? get(function) : getSafe(function).getOrElse(orElseF);
    }
}
