package ru.yandex.chemodan.util.retry;

import net.jodah.failsafe.FailsafeException;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.bolts.collection.Option;

/**
 * @author metal
 */
public class RetryManagerTest {

    @Test
    public void testRunWithRetriesWithoutFailure() {
        TestRunner testRunner = new TestRunner();
        new RetryManager()
                .withLogging("Test retry")
                .withRetryPolicy(10, 100)
                .runSafe(testRunner::run);
        Assert.assertFalse(testRunner.failure);
    }

    @Test
    public void testRunWithRetriesWithFailure() {
        TestRunner testRunner = new TestRunner();
        new RetryManager()
                .withRetryPolicy(2, 100)
                .withFailureCallback(testRunner::failureCallback)
                .runSafe(testRunner::run);
        Assert.assertTrue(testRunner.failure);
    }

    @Test(expected = FailsafeException.class)
    public void testRunUnsafeWithRetriesWithFailure() {
        TestRunner testRunner = new TestRunner();
        new RetryManager()
                .withRetryPolicy(2, 100)
                .withFailureCallback(testRunner::failureCallback)
                .run(testRunner::run);
    }

    @Test
    public void testGetWithRetriesWithoutFailure() {
        TestRunner testRunner = new TestRunner();
        Option<String> result = new RetryManager<String>()
                .withRetryPolicy(7, 100)
                .getSafe(testRunner::get);
        Assert.assertFalse(testRunner.failure);
        Assert.assertEquals("done", result.get());
    }

    @Test
    public void testGetWithRetriesWithFailure() {
        TestRunner testRunner = new TestRunner();
        Option<String> result = new RetryManager<String>()
                .withRetryPolicy(3, 100)
                .withFailureCallback(testRunner::failureCallback)
                .getSafe(testRunner::get);
        Assert.assertFalse(result.isPresent());
        Assert.assertTrue(testRunner.failure);
    }

    @Test(expected = FailsafeException.class)
    public void testGetUnsafeWithRetriesWithFailure() {
        TestRunner testRunner = new TestRunner();
        new RetryManager<String>()
                .withRetryPolicy(3, 100)
                .withFailureCallback(testRunner::failureCallback)
                .get(testRunner::get);
    }

    private static class TestRunner {
        private int currentAttempt = 1;
        public boolean failure = false;

        public void run() {
            if (currentAttempt < 6) {
                currentAttempt++;
                throw new TestException("retries limit");
            }
        }

        public String get() {
            if (currentAttempt < 6) {
                currentAttempt++;
                throw new TestException("retries limit");
            }
            return "done";
        }

        public void failureCallback() {
            failure = true;
        }
    }

    private static class TestException extends RuntimeException {
        public TestException(String message) {
            super(message);
        }
    }
}
