package ru.yandex.chemodan.util.retry;

import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import org.apache.commons.lang3.mutable.MutableObject;
import org.apache.http.HttpException;
import org.joda.time.Duration;

import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.log.LoggerProxies;
import ru.yandex.chemodan.log.TskvNdcUtil;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.ndc.Ndc;

/**
 * @author dbrylev
 */
public class RetryProxy {

    public static <T> T proxy(T target, Class<T> iface,
            int retryCount, Function1B<Throwable> shouldRetry)
    {
        return proxy(target, iface, retryCount, shouldRetry, Interceptor.NO_OP);
    }

    public static <T, I> T proxy(T target, Class<T> iface,
            int retryCount, Function1B<Throwable> shouldRetry, Interceptor<I> interceptor)
    {
        return proxy(target, iface, LoggerProxies.noStackTrace(iface),
                retryCount, Duration.ZERO, 0, shouldRetry, interceptor);
    }

    @SuppressWarnings("unchecked")
    public static <T, I> T proxy(T target, Class<T> iface,
            Logger logger, int retryCount, Duration delay, double delayMultiplier,
            Function1B<Throwable> shouldRetry, Interceptor<I> interceptor)
    {
        InvocationHandler handler = new RetryInvocationHandler(
                target, logger, retryCount, (int) delay.getMillis(), delayMultiplier,
                shouldRetry.notF(), interceptor);

        return (T) Proxy.newProxyInstance(RetryProxy.class.getClassLoader(), new Class<?>[] { iface }, handler);
    }

    public static Interceptor<Ndc.Handle> pushToNdcInterceptor() {
        return new Interceptor<Ndc.Handle>() {
            @Override
            public Ndc.Handle before(Invocation invocation) {
                return invocation.attempt > 0
                        ? TskvNdcUtil.pushToNdc("attempt", invocation.attempt)
                        : TskvNdcUtil.pushToNdc("method", invocation.method.getName());
            }

            @Override
            public void after(Ndc.Handle tag) {
                tag.popSafely();
            }
        };
    }

    private static class RetryInvocationHandler<I> implements InvocationHandler {
        private final Object target;

        private final Logger logger;
        private final int retryCount;

        private final int delayMillis;
        private final double delayMultiplier;

        private final Function1B<Throwable> shouldNotRetry;
        private final Interceptor<I> interceptor;

        public RetryInvocationHandler(
                Object target, Logger logger, int retryCount, int delayMillis, double delayMultiplier,
                Function1B<Throwable> shouldNotRetry, Interceptor<I> interceptor)
        {
            this.target = target;
            this.logger = logger;
            this.retryCount = retryCount;
            this.delayMillis = delayMillis;
            this.delayMultiplier = delayMultiplier;
            this.shouldNotRetry = shouldNotRetry;
            this.interceptor = interceptor;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            MutableObject<Invocation> invocation = new MutableObject<>(new Invocation(method, args, 1));

            Function0 action = () -> {
                I tag = interceptor.before(invocation.getValue());
                try {
                    return method.invoke(this.target, args);

                } catch (InvocationTargetException | IllegalAccessException e) {
                    throw ExceptionUtils.throwException(e.getCause());

                } finally {
                    interceptor.after(tag);
                    invocation.setValue(invocation.getValue().withIncAttempt());
                }
            };

            I tag = interceptor.before(new Invocation(method, args, 0));
            try {
                return RetryUtils.retryOrThrow(logger, retryCount, delayMillis, delayMultiplier, action, shouldNotRetry);
            } finally {
                interceptor.after(tag);
            }
        }
    }

    public interface Interceptor<T> {
        Interceptor<Void> NO_OP = new Interceptor<Void>() {};

        default T before(Invocation invocation) { return null; }

        default void after(T tag) { }
    }

    public static class Invocation {
        public final Method method;
        public final Object[] args;
        public final int attempt;

        public Invocation(Method method, Object[] args, int attempt) {
            this.method = method;
            this.args = args;
            this.attempt = attempt;
        }

        public Invocation withIncAttempt() {
            return new Invocation(method, args, attempt + 1);
        }
    }

    public interface Clauses {

        Function1B<Throwable> http =
                t -> t instanceof HttpException || t instanceof IOException || t instanceof RuntimeIoException;
    }
}
