package ru.yandex.chemodan.util.servicemap;

import javax.annotation.PostConstruct;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.jdbc.JdbcDatabaseConfigurator;
import ru.yandex.chemodan.util.sharpei.ShardInfoSource;
import ru.yandex.chemodan.util.sharpei.SharpeiShardInfoSource;
import ru.yandex.chemodan.util.sharpei.StaticShardInfoSource;
import ru.yandex.misc.spring.ApplicationContextUtils;
import ru.yandex.misc.spring.Service;

/**
 * @author tolmalev
 */
public class DataSourceCollector implements ApplicationContextAware, Service {
    private ApplicationContext applicationContext;
    private ListF<JdbcDatabaseConfigurator> dataSources = Cf.list();

    @PostConstruct
    public void init() {
        dataSources = ApplicationContextUtils.beansOfType(applicationContext, JdbcDatabaseConfigurator.class);
    }

    public ListF<JdbcComponentConnection> getConnections() {
        return dataSources.map(configurator -> {
            String name = configurator.getDataSourceProperties().getPropertyPrefix().toJavaIdentifier();

            JdbcComponentConnection.Type type;
            Option<String> sharpeiHostO = Option.empty();
            ListF<JdbcComponentConnection.ShardInfo> shards = Cf.list();

            if (configurator instanceof JdbcDatabaseConfigurator.DynamicConfigurator) {
                ShardInfoSource shardInfoSource = ((JdbcDatabaseConfigurator.DynamicConfigurator) configurator).getShardInfoSource();
                if (shardInfoSource instanceof SharpeiShardInfoSource) {
                    type = JdbcComponentConnection.Type.SHARPEI;
                    sharpeiHostO = Option.of(((SharpeiShardInfoSource) shardInfoSource).getSharpeiBaseUrl());
                } else if (shardInfoSource.getClass().getSimpleName().equals("ZkShardSource")) {
                    type = JdbcComponentConnection.Type.ZK_DYNAMIC;
                } else if (shardInfoSource instanceof StaticShardInfoSource) {
                    type = JdbcComponentConnection.Type.STATIC;
                } else {
                    type = JdbcComponentConnection.Type.UNKNOWN;
                }

                shards = shardInfoSource.getShards().map(si -> new JdbcComponentConnection.ShardInfo(si.getId(), si.getName(),
                        si.getReaderHostPortDbnames().plus(si.getWriterHostPortDbname()).unique().toList())
                );
            } else {
                type = JdbcComponentConnection.Type.STATIC;

                shards = configurator.getConfiguredHostPortDbNames()
                        .map(hostPortDbNames -> new JdbcComponentConnection.ShardInfo(1, "single", hostPortDbNames));
            }

            return new JdbcComponentConnection(
                    name,
                    type,
                    sharpeiHostO,
                    shards
            );
        });
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void start() throws Exception {
        init();
    }

    @Override
    public void stop() throws Exception {
    }
}
