package ru.yandex.chemodan.util.sharpei;

import java.util.concurrent.CountDownLatch;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.util.ping.PingerChecker;
import ru.yandex.commune.db.shard.ShardInfo;
import ru.yandex.commune.db.shard2.ShardMetaNotifier;
import ru.yandex.commune.db.shard2.ShardMetaWatcher;
import ru.yandex.misc.concurrent.CountDownLatches;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author tolmalev
 */
public class DynamicShardMetaNotifier extends DelayingWorkerServiceBeanSupport implements ShardMetaNotifier,
        PingerChecker {
    private final static Logger logger = LoggerFactory.getLogger(DynamicShardMetaNotifier.class);

    private final ShardInfoSource shardInfoSource;
    private final CountDownLatch initializedLatch = new CountDownLatch(1);

    private Option<ShardMetaWatcher> watcher = Option.empty();

    private ListF<ShardInfo> shards = Cf.arrayList();

    public DynamicShardMetaNotifier(ShardInfoSource shardInfoSource) {
        this.shardInfoSource = shardInfoSource;
    }

    @Override
    public void registerWatcherAndInvoke(ShardMetaWatcher watcher) {
        awaitInitialization();

        watcher.changed(getShards(), Cf.list(), Cf.list());
        this.watcher = Option.of(watcher);
    }

    @Override
    public String getShardTypeName() {
        return "sharpei-shard";
    }

    public boolean isInitialized() {
        return initializedLatch.getCount() == 0;
    }

    protected void checkInitialized() {
        Check.isTrue(isInitialized(), "Not initialized");
    }

    protected void awaitInitialization() {
        CountDownLatches.await(initializedLatch);
    }

    private void initialized() {
        initializedLatch.countDown();
    }

    public ListF<ShardInfo> getShards() {
        return shards;
    }

    @Override
    protected void execute() throws Exception {
        ListF<ShardInfo> newShards = shardInfoSource.getShards();

        logger.debug("get from sharpei-client: {}", newShards);

        MapF<Integer, ShardInfo> nowById = shards.toMapMappingToKey(ShardInfo::getId);

        SetF<Integer> newIds = newShards.map(ShardInfo::getId).unique();

        watcher.forEach(w -> {
            ListF<ShardInfo> removedShards = shards.filter(shard -> !newIds.containsTs(shard.getId()));
            ListF<ShardInfo> changed = newShards.filter(
                    // it's a new shard or shard info has changed
                    shard -> !nowById.containsKeyTs(shard.getId()) || shardChanged(nowById.getTs(shard.getId()), shard)
            );

            if (changed.isNotEmpty() || removedShards.isNotEmpty()) {
                logger.info("about to update shards info all: {}, changed: {}, removedShards: {}",
                        newShards, changed, removedShards);
                w.changed(newShards, changed, removedShards);
            }
        });

        shards = newShards;
        initialized();
    }

    public static boolean shardChanged(ShardInfo oldShard, ShardInfo newShard) {
        return !getShardHostportDbNamesMasterLast(oldShard).equals(getShardHostportDbNamesMasterLast(newShard));
    }

    private static ListF<String> getShardHostportDbNamesMasterLast(ShardInfo oldShard) {
        return oldShard.getReaderHostPortDbnames()
                .sorted()
                .plus(Option.ofNullable(StringUtils.trimToNull(oldShard.getWriterHostPortDbname())));
    }

    @Override
    protected Duration defaultDelay() {
        return Duration.standardSeconds(1);
    }

    @Override
    protected boolean defaultSleepBeforeFirstRun() {
        return false;
    }

    @Override
    public boolean isActive() {
        return isInitialized();
    }
}
