package ru.yandex.chemodan.util.sharpei;

import java.util.Optional;
import java.util.function.Supplier;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.cache.TimeLimitedCacheFast;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class SharpeiCachingManager extends SharpeiManager {
    private final static Logger logger = LoggerFactory.getLogger(SharpeiCachingManager.class);

    private final TimeLimitedCacheFast<String, ShardUserInfo> cache;
    public static final DynamicProperty<Integer> ttl = new DynamicProperty<>("sharpei.client-cache-ttl-minutes", 1);

    public SharpeiCachingManager(Function0<Integer> ttl, SharpeiClientWithRetries sharpeiClient) {
        super(sharpeiClient);

        this.cache = TimeLimitedCacheFast.<String, ShardUserInfo>builder()
                .cacheName("sharpei-users")
                .initialCapacity(2^16)
                .loadFactor(0.6f)
                .ttl(ttl)
                .build();

        logger.info("Cache enabled with ttl: {}", ttl);
    }

    public SharpeiCachingManager(SharpeiClientWithRetries sharpeiClient) {
        this(ttl::get, sharpeiClient);
    }

    @Override
    public Option<ShardUserInfo> find(UserId user, boolean forRead) {
        Optional<ShardUserInfo> fromCache = cache.getFromCache(user.asString());
        if (fromCache.isPresent()) {
            return Option.x(fromCache);
        }

        Option<ShardUserInfo> fromSharpei = sharpeiClient.findOrGetUserShard(user, fromCacheSupplier(user));
        if (fromSharpei.isPresent() && !fromSharpei.get().isRo()) {
            cache.putInCache(user.asString(), fromSharpei.get());
        }

        if (!fromSharpei.isPresent()) {
            return forRead ? Option.empty() : Option.of(registerIfNotExists(user));
        }

        return fromSharpei;
    }

    public ShardUserInfo registerIfNotExists(UserId user) {
        Option<ShardUserInfo> createUser = sharpeiClient.createOrGetUserShard(user, fromCacheSupplier(user));
        createUser.forEach(info -> cache.putInCache(user.asString(), info));
        return createUser.map(info -> info).get();
    }

    @Override
    protected void updateReadOnlyInner(UserId user, SharpeiUserInfo info, boolean readOnly) {
        super.updateReadOnlyInner(user, info, readOnly);
        cache.removeFromCache(user.asString());
    }

    @Override
    protected void updateShardIdAndReadOnlyInner(UserId user, SharpeiUserInfo info, int toShardId, boolean readOnly) {
        super.updateShardIdAndReadOnlyInner(user, info, toShardId, readOnly);
        cache.removeFromCache(user.asString());
    }

    @Override
    protected void updateMigratedInner(UserId user, SharpeiUserInfo info, String ref, boolean migrated) {
        super.updateMigratedInner(user, info, ref, migrated);
        cache.removeFromCache(user.asString());
    }

    private Supplier<Option<ShardUserInfo>> fromCacheSupplier(UserId user) {
        return () -> Option.x(cache.getFromCache(user.asString()));
    }

    public long getCacheSize() {
        return cache.size();
    }

    public void flushCache() {
        cache.flush();
    }
}
