package ru.yandex.chemodan.util.sharpei;

import java.util.concurrent.TimeUnit;

import org.joda.time.Duration;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.test.TestHelper;
import ru.yandex.chemodan.util.retry.RetryConf;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author tolmalev
 */
public class SharpeiClientTest {

    private SharpeiClientWithRetries client = new SharpeiClientWithRetries(
            new SharpeiHttpClient("http://ds-sharpei.dst.yandex.net",
                    ApacheHttpClientUtils.singleConnectionClient(Timeout.timeout(1, TimeUnit.SECONDS))),
            new RetryConf(1, Duration.ZERO), new RetryConf(1, Duration.ZERO));

    static {
        TestHelper.initialize();
    }

    @Test
    public void testMeta() {
        SharpeiUserInfo.Meta meta = new SharpeiUserInfo.Meta();
        Assert.equals("{}", new String(meta.getJsonStringBytes()));
        meta.setMigrated("xxx");
        Assert.equals("{\"m\":[\"xxx\"]}", new String(meta.getJsonStringBytes()));
        meta.cleanMigrated("xxx");
        Assert.equals("{}", new String(meta.getJsonStringBytes()));
    }

    @Test
    public void existing() {
        Assert.some(client.findUser(new SimpleUserId("123")));

        Assert.some(client.findUser(new SimpleUserId("public")));
        Assert.some(client.findUser(new SimpleUserId("yandex_uid:123")));
        Assert.some(client.findUser(new SimpleUserId("device_id:123")));
        Assert.some(client.findUser(new SimpleUserId("uuid:123")));
        Assert.some(client.findUser(new SimpleUserId("yt:123")));
    }

    @Test
    public void nonExisting() {
        Assert.none(client.findUser(new SimpleUserId("232580654437346")));
        Assert.none(client.findUser(new SimpleUserId("yandex_uid:232580654437346")));
        Assert.none(client.findUser(new SimpleUserId("device_id:232580654437346")));
        Assert.none(client.findUser(new SimpleUserId("uuid:232580654437346")));
        Assert.none(client.findUser(new SimpleUserId("yt:232580654437346")));
    }

    @Test
    public void shardsList() {
        Assert.notEmpty(client.getShards());
    }

    @Test
    public void createUser() {
        testCreateUser(() -> new SimpleUserId(Random2.R.nextNonNegativeLong() + ""));
    }

    @Test
    public void createStringUser() {
        testCreateUser(() -> new SimpleUserId(Random2.R.nextAlnum(15)));
    }

    private void testCreateUser(Function0<UserId> genUserIdF) {
        for (Integer i : Cf.range(0, 1000)) {
            UserId uid = genUserIdF.apply();
            if (client.findUser(uid).isPresent()) {
                continue;
            }

            Option<SharpeiUserInfo> created = client.createUser(uid);

            Option<SharpeiUserInfo> existing = Option.empty();
            for (int j = 0; !existing.isPresent() && j < 10; j++) {
                existing = client.findUser(uid);
                if (!existing.isPresent()) {
                    ThreadUtils.sleep(100);
                }
            }

            Assert.equals(created, existing);
            return;
        }
        Assert.fail("");
    }

    @Test
    public void duplicateCreateUser() {
        testDuplicateCreateUser(() -> new SimpleUserId(Random2.R.nextNonNegativeLong() + ""));
    }

    @Test
    public void duplicateCreateStringUser() {
        testDuplicateCreateUser(() -> new SimpleUserId(Random2.R.nextAlnum(15)));
    }

    private void testDuplicateCreateUser(Function0<UserId> genUserIdF) {
        for (Integer i : Cf.range(0, 1000)) {
            UserId uid = genUserIdF.apply();

            if (client.findUser(uid).isPresent()) {
                continue;
            }

            Option<SharpeiUserInfo> created1 = client.createUser(uid);

            ThreadUtils.sleep(Duration.standardSeconds(2));

            Option<SharpeiUserInfo> created2 = client.createUser(uid);

            Assert.equals(created1, created2);
            return;
        }
        Assert.fail("");
    }

    @Test
    public void jsonData() {
        SimpleUserId uid = new SimpleUserId(Random2.R.nextNonNegativeLong() + "");
        client.createUser(uid);

        SharpeiUserInfo.Meta meta = new SharpeiUserInfo.Meta();
        meta.setReadOnly(true);

        client.updateUser(uid, Option.empty(), Option.of(meta));
        Assert.some(meta, client.findUser(uid).map(SharpeiUserInfo::getMeta));

        meta.setReadOnly(false);

        client.updateUser(uid, Option.empty(), Option.of(meta));
        Assert.some(meta, client.findUser(uid).map(SharpeiUserInfo::getMeta));
    }

    @Test
    public void updateShardAndMeta() {
        SimpleUserId uid = new SimpleUserId(Random2.R.nextNonNegativeLong() + "");
        client.createUser(uid);

        int prevShard = -1;
        for (int i = 0; i < 1000; i++) {
            Option<SharpeiUserInfo> userO = client.findUser(uid);
            if (userO.isPresent()) {
                prevShard = userO.get().shard.getId();
                break;
            }
            ThreadUtils.sleep(10);
        }

        int finalPrevShard = prevShard;
        int nextShard = client.getShards().map(SharpeiShardInfo::getId).find(i -> finalPrevShard != i).get();

        client.updateUser(uid, Option.of(Tuple2.tuple(prevShard, nextShard)), Option.empty());
        assertNewShard(uid, nextShard);

        SharpeiUserInfo.Meta meta = new SharpeiUserInfo.Meta();
        meta.setReadOnly(true);

        client.updateUser(uid, Option.of(Tuple2.tuple(nextShard, prevShard)), Option.of(meta));
        assertNewShard(uid, prevShard);
        Assert.some(meta, client.findUser(uid).map(SharpeiUserInfo::getMeta));
    }

    private void assertNewShard(SimpleUserId uid, int expectedShard) {
        Option<Integer> actualShard = Option.empty();
        for (int i = 0; i < 1000; i++) {
            Option<SharpeiUserInfo> userO = client.findUser(uid);
            actualShard = userO.map(u -> u.getShard().getId());
            if (actualShard.isSome(expectedShard)) {
                return;
            }
            ThreadUtils.sleep(10);
        }
        Assert.some(expectedShard, actualShard);
    }
}
