package ru.yandex.chemodan.util.sharpei;

import java.io.IOException;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.FailsafeException;
import net.jodah.failsafe.Listeners;
import net.jodah.failsafe.RetryPolicy;
import net.jodah.failsafe.SyncFailsafe;
import net.jodah.failsafe.function.CheckedRunnable;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.util.retry.RetryConf;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class SharpeiClientWithRetries implements SharpeiClient {

    private static final Logger logger = LoggerFactory.getLogger(SharpeiClientWithRetries.class);

    private final SharpeiClient client;

    private final SyncFailsafe creationFailsafe;
    private final SyncFailsafe defaultFailsafe;

    public SharpeiClientWithRetries(
            SharpeiClient client, RetryConf createUserRetries, RetryConf defaultRetries)
    {
        this.client = client;
        this.creationFailsafe = failsafe(createUserRetries);
        this.defaultFailsafe = failsafe(defaultRetries);
    }

    private static SyncFailsafe failsafe(RetryConf conf) {
        RetryPolicy policy = new RetryPolicy().withMaxRetries(conf.retries);

        if (conf.delay.isLongerThan(Duration.ZERO)) {
            policy = policy.withDelay(conf.delay.getMillis(), TimeUnit.MILLISECONDS);
        }

        policy = policy.retryOn(Cf.list(IOException.class, RuntimeIoException.class));

        Listeners<?> listeners = new Listeners<>().onFailedAttempt((r, e, ctx) ->
                logger.warn("Failed to perform action#{}: {}", ctx.getExecutions(), ExceptionUtils.getAllMessages(e)));

        return Failsafe.with(policy).with(listeners);
    }

    public ListF<SharpeiShardInfo> getShards() {
        return execute(defaultFailsafe, client::getShards);
    }

    public Option<SharpeiUserInfo> findUser(UserId userId) {
        return execute(defaultFailsafe, () -> client.findUser(userId));
    }

    public Option<SharpeiUserInfo> createUser(UserId userId) {
        return execute(creationFailsafe, () -> client.createUser(userId));
    }

    public void updateUser(UserId userId,
            Option<Tuple2<Integer, Integer>> shardUpdate,
            Option<SharpeiUserInfo.Meta> metaUpdate)
    {
        run(defaultFailsafe, () -> client.updateUser(userId, shardUpdate, metaUpdate));
    }

    public Option<ShardUserInfo> createOrGetUserShard(UserId userId, Supplier<Option<ShardUserInfo>> supplier) {
        return execute(creationFailsafe, requestOrGetUser(() -> client.createUser(userId)
                .map(SharpeiUserInfo::toShardUserInfo), supplier));
    }

    public Option<ShardUserInfo> findOrGetUserShard(UserId userId, Supplier<Option<ShardUserInfo>> supplier) {
        return execute(defaultFailsafe, requestOrGetUser(() -> client.findUser(userId)
                .map(SharpeiUserInfo::toShardUserInfo), supplier));
    }

    @Override
    public String getSharpeiBaseUrl() {
        return client.getSharpeiBaseUrl();
    }

    private <T> Callable<Option<T>> requestOrGetUser(Supplier<Option<T>> request, Supplier<Option<T>> supplier) {
        return () -> supplier.get().orElse(request::get).orElse(supplier::get);
    }

    private <T> T execute(SyncFailsafe failsafe, Callable<T> callable) {
        try {
            return failsafe.get(callable);

        } catch (FailsafeException e) {
            throw ExceptionUtils.throwException(e.getCause());
        }
    }

    private void run(SyncFailsafe failsafe, CheckedRunnable runnable) {
        execute(failsafe, () -> { runnable.run(); return null; });
    }
}
