package ru.yandex.chemodan.util.sharpei;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;

import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.bender.parse.JacksonJsonNodeWrapper;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

/**
 * @author tolmalev
 */
public class SharpeiHttpClient implements SharpeiClient {
    private final String baseUrl;
    private final HttpClient httpClient;

    private static final BenderJsonParser<SharpeiShardInfo> shardInfoParser = Bender.jsonParser(SharpeiShardInfo.class);

    private static final BenderJsonParser<SharpeiUserInfo> userInfoParser = Bender.jsonParser(SharpeiUserInfo.class);

    public SharpeiHttpClient(String baseUrl, HttpClient httpClient) {
        this.baseUrl = baseUrl;
        this.httpClient = httpClient;
    }

    public ListF<SharpeiShardInfo> getShards() {
        URI uri = UriBuilder
                .cons(baseUrl)
                .appendPath("stat")
                .build();

        String response = ApacheHttpClientUtils.execute(new HttpGet(uri), httpClient, new StringResponseHandler());
        ObjectNode node = (ObjectNode) JsonNodeUtils.getNode(response);

        return Cf.x(node.fieldNames())
                .toList()
                .map(shardId -> shardInfoParser.parseJson(new JacksonJsonNodeWrapper(node.get(shardId))));
    }

    public void updateUser(UserId userId,
            Option<Tuple2<Integer, Integer>> shardUpdate,
            Option<SharpeiUserInfo.Meta> metaUpdate)
    {
        String stringUid = userId.asString();

        URI uri = UriBuilder
                .cons(baseUrl)
                .appendPath("update_user")
                .addParam("uid", stringUid)
                .addParams(shardUpdate
                        .map(t -> Tuple2List.fromPairs("shard_id", t.get1() + "", "new_shard_id", t.get2() + ""))
                        .getOrElse(Tuple2List::tuple2List))
                .build();

        HttpPost post = new HttpPost(uri);
        post.setHeader("Content-Type", "application/json");
        metaUpdate.forEach(m -> post.setEntity(new ByteArrayEntity(m.getJsonStringBytes())));

        ApacheHttpClientUtils.execute(post, httpClient, new StringResponseHandler());
    }

    public Option<SharpeiUserInfo> createUser(UserId userId) {
        String stringUid = userId.asString();

        URI uri = UriBuilder
                .cons(baseUrl)
                .appendPath("create_user")
                .addParam("uid", stringUid)
                .build();

        return ApacheHttpClientUtils.execute(new HttpPost(uri), httpClient,
                response -> {
                    int statusCode = response.getStatusLine().getStatusCode();
                    if (statusCode == HttpStatus.SC_NOT_FOUND) {
                        return Option.empty();
                    }
                    if (statusCode == HttpStatus.SC_OK || statusCode == HttpStatus.SC_CREATED) {
                        SharpeiShardInfo shardInfo =
                                shardInfoParser.parseJson(InputStreamSourceUtils.wrap(response.getEntity().getContent()));
                        return Option.of(shardInfo);
                    } else {
                        throw new StrangeResponseException(statusCode, response.getEntity().getContent());
                    }
                }
        ).map(shard -> new SharpeiUserInfo((SharpeiShardInfo) shard, Option.empty()));
    }

    public Option<SharpeiUserInfo> findUser(UserId userId) {
        String stringUid = userId.asString();

        URI uri = UriBuilder
                .cons(baseUrl)
                .appendPath("get_user")
                .addParam("uid", stringUid)
                .addParam("mode", "all")
                .addParam("format", "json")
                .build();

        return ApacheHttpClientUtils.execute(new HttpGet(uri), httpClient,
                response -> {
                    int statusCode = response.getStatusLine().getStatusCode();
                    if (statusCode == HttpStatus.SC_NOT_FOUND) {
                        return Option.empty();
                    }
                    String text = InputStreamSourceUtils.wrap(response.getEntity().getContent()).readText();

                    if (statusCode == HttpStatus.SC_OK || statusCode == HttpStatus.SC_CREATED) {
                        SharpeiUserInfo userInfo = userInfoParser.parseJson(text);
                        return Option.of(userInfo);
                    } else {
                        throw new StrangeResponseException(statusCode, text);
                    }
                }
        );
    }

    @Override
    public String getSharpeiBaseUrl() {
        return baseUrl;
    }

    private static class StringResponseHandler implements ResponseHandler<String> {

        public String handleResponse(HttpResponse response) throws IOException {
            int statusCode = response.getStatusLine().getStatusCode();

            if (!ru.yandex.misc.io.http.HttpStatus.is2xx(statusCode)) {
                throw new StrangeResponseException(statusCode, response.getEntity().getContent());
            }
            return InputStreamSourceUtils.wrap(response.getEntity().getContent()).readText();
        }
    }

    private static class StrangeResponseException extends HttpException {

        public StrangeResponseException(int statusCode, InputStream response) {
            super(statusCode, InputStreamSourceUtils.wrap(response).readText());
        }

        public StrangeResponseException(int statusCode, String response) {
            super(statusCode, "Strange response from sharpei: (" + statusCode + "): " + response);
        }
    }
}
