package ru.yandex.chemodan.util.sharpei;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;

/**
 * @author yashunsky
 */
public class SharpeiManager {
    protected final SharpeiClientWithRetries sharpeiClient;

    public SharpeiManager(SharpeiClientWithRetries sharpeiClient) {
        this.sharpeiClient = sharpeiClient;
    }

    public Option<ShardUserInfo> find(UserId user, boolean forRead) {
        Option<ShardUserInfo> info = sharpeiClient.findOrGetUserShard(user, Option::empty);

        if (!info.isPresent()) {
            return forRead ? Option.empty() : Option.of(registerIfNotExists(user));
        }

        return info;
    }

    public Option<ShardUserInfo> find(UserId user) {
        return find(user, false);
    }

    public ListF<ShardUserInfo> find(ListF<UserId> user) {
        return user.flatMap(this::find);
    }

    public ShardUserInfo registerIfNotExists(UserId user) {
        Option<ShardUserInfo> createUser = sharpeiClient.createOrGetUserShard(user, Option::empty);
        return createUser.map(info -> info).get();
    }

    public void registerBatch(ListF<UserId> usersIds) {
        usersIds.forEach(this::registerIfNotExists);
    }

    public void updateReadOnly(UserId user, boolean readOnly) {
        sharpeiClient.findUser(user).ifPresent(info -> updateReadOnlyInner(user, info, readOnly));
    }

    protected void updateReadOnlyInner(UserId user, SharpeiUserInfo info, boolean readOnly) {
        SharpeiUserInfo.Meta meta = info.getMeta();
        meta.setReadOnly(readOnly);
        sharpeiClient.updateUser(user, Option.empty(), Option.of(meta));
    }

    public void updateShardIdAndReadOnly(UserId user, int toShardId, boolean readOnly) {
        sharpeiClient.findUser(user).ifPresent(info -> updateShardIdAndReadOnlyInner(user, info, toShardId, readOnly));
    }

    protected void updateShardIdAndReadOnlyInner(UserId user, SharpeiUserInfo info, int toShardId, boolean readOnly) {
        SharpeiUserInfo.Meta meta = info.getMeta();
        meta.setReadOnly(readOnly);
        sharpeiClient.updateUser(user, Option.of(Tuple2.tuple(info.shard.getId(), toShardId)), Option.of(meta));
    }

    public void updateMigrated(UserId user, String ref, boolean migrated) {
        sharpeiClient.findUser(user).ifPresent(info -> updateMigratedInner(user, info, ref, migrated));
    }

    protected void updateMigratedInner(UserId user, SharpeiUserInfo info, String ref, boolean migrated) {
        SharpeiUserInfo.Meta meta = info.getMeta();
        if (migrated) {
            meta.setMigrated(ref);
        } else {
            meta.cleanMigrated(ref);
        }
        sharpeiClient.updateUser(user, Option.empty(), Option.of(meta));
    }

}
