package ru.yandex.chemodan.util.sharpei;

import java.util.Objects;

import lombok.Data;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.DefaultToString;

/**
 * @author tolmalev
 */
@BenderBindAllFields
@Data
public class SharpeiUserInfo {
    public final SharpeiShardInfo shard;
    private final Option<String> data;

    @BenderIgnore
    private transient Option<Meta> parsedMeta;

    public SharpeiUserInfo(SharpeiShardInfo shard, Option<String> data) {
        this.shard = shard;
        this.data = data;
    }

    @BenderBindAllFields
    public static class Meta extends DefaultToString {
        private static final BenderParserSerializer<Meta> PS = Bender.cons(Meta.class);

        @BenderPart(name = "r")
        private Option<Integer> readOnly = Option.empty();

        @BenderPart(wrapperName = "m")
        private Option<SetF<String>> migrated = Option.empty();

        @Override
        public boolean equals(Object obj) {
            if (!obj.getClass().equals(getClass())) {
                return false;
            }
            Meta meta = (Meta) obj;

            return isReadOnly() == meta.isReadOnly() && this.migrated.equals(meta.migrated);
        }

        @Override
        public int hashCode() {
            return Objects.hash(readOnly, migrated);
        }

        public boolean isMigrated(String db) {
            return migrated.isMatch(s -> s.containsTs(db));
        }

        public void setMigrated(String db) {
            migrated = Option.of(migrated.getOrElse(Cf::set).plus1(db));
        }

        public void cleanMigrated(String db) {
            migrated = migrated.map(s -> s.minus1(db)).filterNot(SetF::isEmpty);
        }

        public boolean isReadOnly() {
            return readOnly.getOrElse(0) == 1;
        }

        public void setReadOnly(boolean readOnly) {
            this.readOnly = readOnly ? Option.of(1) : Option.empty();
        }

        public byte[] getJsonStringBytes() {
            return PS.getSerializer().serializeJson(this);
        }

        public static Meta fromString(String str) {
            return PS.getParser().parseJson(str);
        }
    }

    public Meta getMeta() {
        if (parsedMeta == null) {
            parsedMeta = data.map(Meta::fromString);
        }
        return parsedMeta.getOrElse(Meta::new);
    }

    public ShardUserInfo toShardUserInfo() {
        return new ShardUserInfo(shard.getId(), getMeta().isReadOnly(), getMeta().migrated.getOrElse(Cf::set));
    }
}
