package ru.yandex.chemodan.util.web;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.boot.CoolPingServletContextConfiguration;
import ru.yandex.chemodan.http.DiskJetty;
import ru.yandex.chemodan.log.DiskLog4jRequestLog;
import ru.yandex.chemodan.util.ping.CoolPingServlet;
import ru.yandex.commune.a3.ActionApp;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.version.AppName;
import ru.yandex.misc.web.servlet.SelfMappedFilter;
import ru.yandex.misc.web.servlet.WtdFilter;

/**
 * @author tolmalev
 */
@Configuration
@Import({
        CoolPingServletContextConfiguration.class
})
public class A3JettyContextConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(A3JettyContextConfiguration.class);

    @Autowired(required = false)
    private ActionApp actionApp;

    @Autowired(required = false)
    private A3JettyConfiguration a3ServletsConfig;

    @Autowired(required = false)
    private CustomJettyConfigurator customJettyConfigurator;

    @Autowired(required = false)
    private CustomYcridPrefixResolver customYcridPrefixResolver;

    @Bean
    public DiskJetty a3Jetty(
            AppName appName,
            @Value("${a3.port}")
            IpPort port,
            @Value("${a3.qloud.direct-port:-false}")
            boolean directPortInQloud,
            @Value("${a3.maxThreads}")
            int maxThreads,
            @Value("${a3.minThreads}")
            int minThreads,
            @Value("${a3.maxQueueLength}")
            int maxQueueLength,
            @Value("${a3.acceptQueueSize}")
            String acceptQueueSizeStr,
            @Value("${a3.ycrid_prefix:-}")
            String ycridPrefix,
            @Value("${a3.maxRequestHeadersSize:-}")
            Optional<DataSize> maxRequestHeadersSize,
            @Value("${a3.maxIdleTimeout:-30s}")
            Optional<Duration> maxIdleTimeout,
            @Value("${a3.soLinerTimeout:-30s}")
            Optional<Duration> soLinerTimeout,
            @Value("${a3.selectors:-}")
            Optional<Integer> selectors,
            @Value("${a3.acceptors:-}")
            Optional<Integer> acceptors,
            @Value("${a3.sslSessionCacheSize:-}")
            Optional<Integer> sslSessionCacheSize,
            @Value("${a3.secureRandomAlgorithm:-}")
            Optional<String> secureRandomAlgorithm,
            @Value("${https.port:-}")
            Optional<Integer> httpsPort,
            @Value("${https.keystore.file:-}")
            Optional<String> httpsKeystoreFile,
            @Value("${https.keystore.type:-}")
            Optional<String> httpsKeystoreType,
            @Value("${https.keystore.pass:-}")
            Optional<String> httpsKeystorePass,
            @Value("${https.ciphers:-}")
            String ciphers,
            @Value("${https.excluded.protocols:-SSLv3}")
            String excludedSslProtocols,
            @Value("${https.ssl.timeout:-1}") int sslSessionTimeout,
            @Autowired(required = false) Optional<List<SelfMappedFilter>> additionalFilters,
            @Value("${a3.enableDefaultServlet:-false}")
            boolean enableDefaultA3Servlet,
            CoolPingServlet coolPingServlet) {
        DiskJetty jetty = customYcridPrefixResolver == null
                ? new DiskJetty(ycridPrefix.isEmpty() ? appName.appName() : ycridPrefix)
                : new DiskJetty(customYcridPrefixResolver::getPrefix);

        MapF<String, String> env = Cf.x(System.getenv());
        int realPort = port.getPort();
        if (directPortInQloud) {
            realPort = env.getO("QLOUD_HTTP_PORT").map(Integer::parseInt).getOrElse(port.getPort());
        }

        jetty.setHttpPort(realPort);
        jetty.setMaxThreads(maxThreads);
        jetty.setMinThreads(minThreads);
        jetty.setMaxQueueLength(maxQueueLength);
        jetty.setLookupServletsInContext(false);
        jetty.addFilterMapping("/*", new WtdFilter());

        jetty.setCollectInvocationsByStatus(true);

        maxIdleTimeout.ifPresent(duration -> jetty.setMaxIdleTimeMillis((int) duration.getMillis()));
        soLinerTimeout.ifPresent(duration -> jetty.setSoLinerTimeMillisO((int) duration.getMillis()));
        maxRequestHeadersSize.ifPresent(limit -> jetty.setRequestHeaderSizeO((int) limit.toBytes()));

        selectors.ifPresent(jetty::setSelectors);
        acceptors.ifPresent(jetty::setAcceptors);
        if (httpsPort.isPresent()
                && httpsKeystorePass.isPresent()
                && httpsKeystoreFile.isPresent())
        {
            logger.debug("Https port: {}", httpsPort.get());
            jetty.setHttpsPort(httpsPort.get());

            logger.debug("Https keystore file: {}", httpsKeystoreFile.get());
            jetty.setHttpsKeystoreFile(httpsKeystoreFile.get());

            logger.debug("Https keystore type: {}", httpsKeystoreType.get());
            jetty.setHttpsKeystoreType(httpsKeystoreType.get());

            logger.debug("Https keystore pass: {}", httpsKeystorePass.get());
            jetty.setHttpsKeystorePass(httpsKeystorePass.get());
            jetty.setHttpsSslProtocol("https");

            logger.debug("Https ssl session timeout: {}", sslSessionTimeout);
            jetty.setSslSessionTimeout(sslSessionTimeout);

            logger.debug("ExcludedSslProtocols : {}", Cf.list(StringUtils.split(excludedSslProtocols, ",")));
            jetty.setExcludedSslProtocols(Cf.list(StringUtils.split(excludedSslProtocols, ",")));

            ListF<String> cipherSuites = Cf.list(StringUtils.split(ciphers, ","));
            if (cipherSuites.isNotEmpty()) {
                logger.debug("IncludedCipherSuites: {}", cipherSuites);
                jetty.setIncludedCipherSuites(cipherSuites);
            }
            logger.debug("UseStrongCipherSuites: true");
            jetty.setUseStrongCipherSuites(true);

            logger.debug("sslSessionCacheSize: {}", sslSessionCacheSize);
            sslSessionCacheSize.ifPresent(c -> jetty.setSslSessionCacheSize(c));
            Option<String> randomAlgorithm = Option.x(secureRandomAlgorithm.filter(s -> StringUtils.isNotEmpty(s)));
            logger.debug("secureRandomAlgorithm: {}", randomAlgorithm);
            jetty.setSecureRandomAlgorithm(randomAlgorithm);
        }

        jetty.setRequestLogFactory(DiskLog4jRequestLog.consF());

        if (!acceptQueueSizeStr.isEmpty()) {
            jetty.setAcceptQueueSize(Integer.parseInt(acceptQueueSizeStr));
        }

        if (customJettyConfigurator != null) {
            customJettyConfigurator.configure(jetty);
        } else {
            jetty.addFilterMapping("*", new AddYcridToHeadersFilter());
            additionalFilters.ifPresent(s -> s.stream()
                    .filter(Objects::nonNull)
                    .forEach(additionalFilter -> jetty.addFilterMapping(additionalFilter.getUrlPattern(), additionalFilter)));
            jetty.addServletMapping("/ping/*", coolPingServlet);
            if (actionApp != null) {
                if (enableDefaultA3Servlet) {
                    jetty.addServletMapping("/a3/*", actionApp.createServlet());
                }

                if (a3ServletsConfig != null) {
                    a3ServletsConfig.pathToNamespaceList.forEach(
                            (prefix, ns) -> jetty.addServletMapping(prefix, actionApp.createServlet(ns))
                    );

                    a3ServletsConfig.postConstruct(jetty, actionApp);
                }
            }
        }

        return jetty;
    }
}
