package ru.yandex.chemodan.util.yasm;

import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.misc.monica.core.Metered;
import ru.yandex.misc.monica.core.blocks.AverageData;
import ru.yandex.misc.monica.core.blocks.InstrumentedData;
import ru.yandex.misc.monica.core.blocks.StatisticData;
import ru.yandex.misc.monica.core.name.FullMetricName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.monica.core.snapshot.Snapshot;
import ru.yandex.misc.monica.core.stat.quantile.QuantileValues;

/**
 * @author Maksim Leonov
 */
@ParametersAreNonnullByDefault
public class MonicaSnapshotAggregatorForYasm implements YasmSensorsDataProducer {

    private static final int normalize_coeff = 10_000;

    private final MonicaYasmConfiguration configuration;

    private final MapF<Class, ListF<UnistatSensorExtractor>> sensorExtractors;

    private volatile Snapshot mySnapshot = Snapshot.EMPTY;

    public MonicaSnapshotAggregatorForYasm(MonicaYasmConfiguration configuration) {
        this.configuration = configuration;
        this.sensorExtractors = Cf.hashMap();
        addDefaultSensorExtractors();
    }

    public MonicaSnapshotAggregatorForYasm(MonicaYasmConfiguration configuration,
            MapF<Class, ListF<UnistatSensorExtractor>> sensorExtractors)
    {
        this.configuration = configuration;
        this.sensorExtractors = sensorExtractors;
    }

    public void storeSnapshot(Snapshot mySnapshot) {
        this.mySnapshot = mySnapshot;
    }

    @Override
    public UnistatSensorsData getSensors() {
        try {
            return new UnistatSensorsData(convert(mySnapshot, configuration));
        } catch (RuntimeException e) {
            // TODO: error? Or zero data?
            return new UnistatSensorsData(Collections.emptyList());
        }
    }

    public List<UnistatSensor> convert(Snapshot snapshot, MonicaYasmConfiguration configuration) {
        Function1B<FullMetricName> metricFilter = configuration.getMetricFilter();

        return snapshot.getData().entries()
                .filterBy1(metricFilter)
                .flatMap(
                        entry -> {
                            FullMetricName fullName = entry._1;
                            Metered<?> metered = entry._2;

                            MetricName localName = fullName.name;
                            Object value = metered.getValue();

                            return extractSensors(localName, value);
                        }
                );
    }

    public <T> void addSensorExtractor(Class<T> clazz, UnistatSensorExtractor<T> extractor) {
        synchronized (sensorExtractors) {
            sensorExtractors.getOrElseUpdate(clazz, Cf::arrayList).add(extractor);
        }
    }

    public void addDefaultSensorExtractors() {
        Cf.list(Integer.class, Long.class, Double.class).forEach(numberClass ->
                addSensorExtractor(numberClass, (name, value) ->
                        Cf.list(new SimpleUnistatSensor(
                                metricNameToYasmName(name),
                                value.doubleValue(),
                                UnistatSensor.AggregationType.AVERAGE_VALUE))
                )
        );

        addSensorExtractor(StatisticData.class,
                (name, value) -> sensorsForStatistics(value, metricNameToYasmName(name))
        );

        addSensorExtractor(AverageData.class,
                (name, value) -> avgSensorsForAverage(value, metricNameToYasmName(name))
                        .plus(totalSensorsForAverage(value, metricNameToYasmName(name) + "_total"))
        );

        addSensorExtractor(InstrumentedData.class, (name, value) -> {
            String baseName = metricNameToYasmName(name);

            return totalSensorsForAverage(value.errorRate(), baseName + "_errors")
                    .plus(sensorsForStatistics(value.statisticData(), baseName));
        });
    }

    static ListF<UnistatSensor> totalSensorsForAverage(AverageData value, String metricName) {
        return Cf.list(new SimpleUnistatSensor(
                metricName,
                value.getTotal(),
                UnistatSensor.AggregationType.SUMM
        ));
    };

    static ListF<UnistatSensor> avgSensorsForAverage(AverageData value, String baseName) {
        return Cf.list(
                new SimpleUnistatSensor(
                        baseName + "_1min",
                        value.getAverage1Min(),
                        UnistatSensor.AggregationType.AVERAGE_VALUE
                ),
                new SimpleUnistatSensor(
                        baseName + "_5min",
                        value.getAverage5Min(),
                        UnistatSensor.AggregationType.AVERAGE_VALUE
                ),
                new SimpleUnistatSensor(
                        baseName + "_15min",
                        value.getAverage15Min(),
                        UnistatSensor.AggregationType.AVERAGE_VALUE
                )
        );
    }

    static ListF<UnistatSensor> sensorsForStatistics(StatisticData value, String baseName) {
        double rate1Second = value.getMeter().getAverage1Min() / 60;

        ListF<QuantileValues.QuantileValue> qList = value.getQuantiles().asList().sortedBy(qv -> qv.quantile);

        ListF<Tuple2<Double, Long>> histogramData = Cf.arrayList();

        double histValue = qList.get(0).value;
        double histWeight = 0;

        for (int i = 1; i < qList.length(); i++) {
            histWeight += qList.get(i).quantile - qList.get(i - 1).quantile;

            if (qList.get(i).value - histValue > 0.001) {
                histogramData.add(Tuple2.tuple(histValue, Math.round(histWeight * rate1Second * normalize_coeff)));

                histValue = qList.get(i).value;
                histWeight = 0;
            }
        }
        if (histWeight > 0)
            histogramData.add(Tuple2.tuple(histValue, Math.round(histWeight * rate1Second * normalize_coeff)));

        return totalSensorsForAverage(value.getMeter(), baseName + "_rate")
                .plus(new HistUnistatSensor(
                        baseName + "_times",
                        UnistatSensor.AggregationType.ABSOLUTE_HGRAM,
                        histogramData
                ));
    }

    public static String metricNameToYasmName(MetricName localName) {
        return localName.asList().mkString("_");
    }

    private ListF<UnistatSensor> extractSensors(MetricName localName, Object value) {
        return sensorExtractors
                .getOrElse(value.getClass(), Cf.list())
                .flatMap(e -> e.extract(localName, value));
    }

    private interface UnistatSensorExtractor<T> {
        ListF<UnistatSensor> extract(MetricName metricName, T o);
    }
}
