package ru.yandex.chemodan.util.yasm.monitor;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;

/**
 * @author yashunsky
 */
public class MetricAverage {
    private final Duration maxInterval;
    private Instant lastCleaned;
    private ListF<Point> points;

    public MetricAverage(Duration maxInterval) {
        this.maxInterval = maxInterval;
        this.lastCleaned = Instant.now();
        this.points = Cf.arrayList();
    }

    public void update(double value) {
        cleanHead();
        points.add(new Point(value));
    }

    public synchronized Double getAverage(Duration interval,
            Function<Option<Instant>, RuntimeException> onOutOfDateData)
    {
        cleanHead();
        Instant deadline = Instant.now().minus(interval);
        ListF<Double> values = points.filterNot(p -> p.isBefore(deadline)).map(Point::getValue);

        if (values.isEmpty()) {
            Option<Instant> latestPoint = points.map(Point::getTimestamp).maxO();
            throw onOutOfDateData.apply(latestPoint);
        }
        return values.sum(Cf.Double) / values.size();
    }

    private void cleanHead() {
        Instant deadline = Instant.now().minus(maxInterval);
        if (lastCleaned.isBefore(deadline)) {
            points.removeIf(p -> p.isBefore(deadline));
        }
    }

    public int getPointsCountForTest() {
        return points.size();
    }

    @Data
    private static class Point {
        private final Instant timestamp;
        private final double value;

        public Point(double value) {
            this.timestamp = Instant.now();
            this.value = value;
        }

        public boolean isBefore(Instant limit) {
            return timestamp.isBefore(limit);
        }
    }
}
