package ru.yandex.chemodan.util.yasm.monitor;

import java.io.ByteArrayInputStream;
import java.io.IOException;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpUriRequest;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
public class YasmAggrClientTest {
    private final static String MOCK_RESPONSE = "{\n" +
            "    \"aggr\": [\n" +
            "        [\n" +
            "            \"common\",\n" +
            "            \"|ctype,geo,prj,tier\",\n" +
            "            {\n" +
            "                \"iostat-max_load_xhhh\": [\n" +
            "                    [\n" +
            "                        22.899444787168413\n" +
            "                    ],\n" +
            "                    0,\n" +
            "                    null\n" +
            "                ]," +
            "                \"cpu-us_hgram\": [\n" +
            "                    \"ugram\",\n" +
            "                    [\n" +
            "                        [\n" +
            "                            10,\n" +
            "                            1\n" +
            "                        ],\n" +
            "                        [\n" +
            "                            15,\n" +
            "                            0\n" +
            "                        ]\n" +
            "                    ]\n" +
            "                ]\n" +
            "            }\n" +
            "        ],\n" +
            "        [\n" +
            "            \"mailpostgresql\",\n" +
            "            \"ctype=mail-pg-common-test02;geo=sas;prj=pg-common-test02;tier=primary\",\n" +
            "            {\n" +
            "                \"pgbouncer-bouncer_total_avg_query_time_xxxv\": 0.318\n" +
            "            }\n" +
            "        ],\n" +
            "        [\n" +
            "            \"mailpostgresql\",\n" +
            "            \"ctype=mail-pg-common-test02;geo=sas;prj=pg-common-test02|tier\",\n" +
            "            {\n" +
            "                \"pgbouncer-bouncer_total_avg_query_time_xxxv\": 0.318\n" +
            "            }\n" +
            "        ],\n" +
            "        [\n" +
            "            \"mailpostgresql\",\n" +
            "            \"|ctype,geo,prj,tier\",\n" +
            "            {\n" +
            "                \"pgbouncer-bouncer_total_avg_query_time_xxxv\": 0.318\n" +
            "            }\n" +
            "        ],\n" +
            "        [" +
            "            \"mailpostgresql\",\n" +
            "            \"ctype=disk-dataapi-db-04;geo=myt;prj=dataapidb04;tier=replica\",\n" +
            "            {\n" +
            "                \"postgresql-replication_lag_mmmv\": 0\n" +
            "            }\n" +
            "        ]\n" +
            "    ],\n" +
            "    \"status\": \"ok\"\n" +
            "}";

    @Test
    public void request() throws IOException {
        YasmAggrClient client = setupMocks(MOCK_RESPONSE);
        MapF<YasmMetric, Double> result = client.get("mock.host", YasmMetricExtractor.ALL);

        Assert.hasSize(4, result);
        Assert.equals(22.899, result.getTs(YasmMetric.IO), 0.001);
        Assert.equals(0.318, result.getTs(YasmMetric.PG_QUERY_TIME), 0.001);
        Assert.equals(0, result.getTs(YasmMetric.PG_LAG), 0.001);
        Assert.equals(12.5, result.getTs(YasmMetric.CPU_USER), 0.001);
    }

    @Test
    public void testSmartcacheDbRealMetrics() throws IOException {
        InputStreamSource source = new ClassPathResourceInputStreamSource(getClass(), "smartcache_db_metrics.json");
        YasmAggrClient client = setupMocks(new String(source.readBytes()));
        MapF<YasmMetric, Double> result = client.get("mock.host", YasmMetricExtractor.ALL);

        Assert.hasSize(8, result);
        Assert.equals(27.46444335, result.getTs(YasmMetric.IO), 0.001);
        Assert.equals(2.5015, result.getTs(YasmMetric.PG_QUERY_TIME), 0.001);
        Assert.equals(1.0, result.getTs(YasmMetric.CPU_USER), 0.001);
        Assert.equals(0.0, result.getTs(YasmMetric.PG_LAG), 0.001);
        Assert.equals(1.4593788E7, result.getTs(YasmMetric.NETWORK_IN), 0.001);
        Assert.equals(2.25, result.getTs(YasmMetric.LA), 0.001);
        Assert.equals(9.1795173E7, result.getTs(YasmMetric.NETWORK_OUT), 0.001);
        Assert.equals(1292604.0, result.getTs(YasmMetric.PG_DEAD_TUP), 0.001);
    }

    @Test
    public void testDataapiDbRealMetrics() throws IOException {
        InputStreamSource source = new ClassPathResourceInputStreamSource(getClass(), "dataapi_db_metrics.json");
        YasmAggrClient client = setupMocks(new String(source.readBytes()));
        MapF<YasmMetric, Double> result = client.get("mock.host", YasmMetricExtractor.ALL);

        Assert.hasSize(8, result);
        Assert.equals(54.10725552, result.getTs(YasmMetric.IO), 0.001);
        Assert.equals(1.4215, result.getTs(YasmMetric.PG_QUERY_TIME), 0.001);
        Assert.equals(7.0, result.getTs(YasmMetric.CPU_USER), 0.001);
        Assert.equals(0.0, result.getTs(YasmMetric.PG_LAG), 0.001);
        Assert.equals(1.0365079E7, result.getTs(YasmMetric.NETWORK_IN), 0.001);
        Assert.equals(6.17, result.getTs(YasmMetric.LA), 0.001);
        Assert.equals(3.9445734E7, result.getTs(YasmMetric.NETWORK_OUT), 0.001);
        Assert.equals(261650.0, result.getTs(YasmMetric.PG_DEAD_TUP), 0.001);
    }


    private YasmAggrClient setupMocks(String hostResponse) throws IOException {
        HttpClient httpClient = Mockito.mock(HttpClient.class);
        HttpEntity entity = Mockito.mock(HttpEntity.class);
        HttpResponse response = Mockito.mock(HttpResponse.class);

        Mockito.when(entity.getContentLength()).thenReturn((long) hostResponse.length());
        Mockito.when(entity.getContent()).thenReturn(new ByteArrayInputStream(hostResponse.getBytes()));
        Mockito.when(response.getEntity()).thenReturn(entity);

        Mockito.when(httpClient.execute(Mockito.any(HttpUriRequest.class), Mockito.any(ResponseHandler.class)))
                .then(invocation -> {
                    ResponseHandler handler = (ResponseHandler) invocation.getArguments()[1];
                    return handler.handleResponse(response);
                });

        return new YasmAggrClient(httpClient);
    }
}
