package ru.yandex.chemodan.util.yasm.monitor;

import com.fasterxml.jackson.databind.JsonNode;
import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.util.json.JsonNodePathNavigator;
import ru.yandex.chemodan.util.json.SimpleJsonPathNavigator;
import ru.yandex.chemodan.util.json.SuffixJsonPathNavigator;

/**
 * @author yashunsky
 */

@AllArgsConstructor
public class YasmMetricExtractor {
    private final YasmMetric metric;
    private final ListF<JsonNodePathNavigator> path;
    private final Function<JsonNode, Double> parse;

    public Option<Double> extract(JsonNode node) {
        try {
            Option<JsonNode> nodeO = Option.of(node);
            for (JsonNodePathNavigator pathNavigator : path) {
                nodeO = nodeO.flatMapO(pathNavigator::getO);
            }

            return nodeO.map(parse);
        } catch (NullPointerException e) {
            return Option.empty();
        }
    }

    public String getName() {
        return metric.name();
    }
    public YasmMetric getMetric() {
        return metric;
    };

    public static final ListF<YasmMetricExtractor> ALL =
            Cf.list(io(), networkIn(), networkOut(), la(), pgQueryTime(), cpuUser(), pgLag(), pgDeadTup());

    public static final ListF<YasmMetricExtractor> SHARPERI_CLEANING_EXTRACTORS =
            Cf.list(io(), networkIn(), networkOut(), la(), pgQueryTime(), cpuUser(), pgLag());

    public static ListF<JsonNodePathNavigator> simplePath(ListF<String> path) {
        return path.map(SimpleJsonPathNavigator::new);
    }

    public static YasmMetricExtractor io() {
        return new YasmMetricExtractor(YasmMetric.IO,
                simplePath(Cf.list("common", "iostat-max_load_xhhh")),
                node -> node.get(0).get(0).asDouble());
    }

    public static YasmMetricExtractor networkIn() {
        return new YasmMetricExtractor(YasmMetric.NETWORK_IN,
                simplePath(Cf.list("common", "netstat-ibytes_summ")),
                JsonNode::asDouble);
    }

    public static YasmMetricExtractor networkOut() {
        return new YasmMetricExtractor(YasmMetric.NETWORK_OUT,
                simplePath(Cf.list("common", "netstat-obytes_summ")),
                JsonNode::asDouble);
    }

    public static YasmMetricExtractor la() {
        return new YasmMetricExtractor(YasmMetric.LA,
                simplePath(Cf.list("common", "loadavg-abs_xhhh")),
                node -> node.get(0).get(0).asDouble());
    }

    public static YasmMetricExtractor cpuUser() {
        return new YasmMetricExtractor(YasmMetric.CPU_USER,
                simplePath(Cf.list("common", "cpu-us_hgram")),
                node -> {
                    if (!(node.get(0).asText().equals("ugram") && node.get(1).isArray())) {
                        return -1.0;
                    }
                    ListF<Double> values = Cf.x(node.get(1).elements()).toList().map(n -> n.get(0).asDouble());
                    if (values.isEmpty()) {
                        return 0.0;
                    }
                    return values.sum(Cf.Double)/values.size();
                });
    }

    public static YasmMetricExtractor pgQueryTime() {
        return new YasmMetricExtractor(YasmMetric.PG_QUERY_TIME,
                simplePath(Cf.list("mailpostgresql", "pgbouncer-bouncer_total_avg_query_time_xxxv")),
                JsonNode::asDouble);
    }

    public static YasmMetricExtractor pgLag() {
        return new YasmMetricExtractor(YasmMetric.PG_LAG,
                simplePath(Cf.list("mailpostgresql", "postgresql-replication_lag_mmmv")),
                node -> node.asDouble() / 5);
    }

    public static YasmMetricExtractor pgDeadTup() {
        return new YasmMetricExtractor(YasmMetric.PG_DEAD_TUP,
                Cf.list(new SimpleJsonPathNavigator("mailpostgresql"),
                        new SuffixJsonPathNavigator("stat_dead_tup_tmmv")),
                JsonNode::asDouble);
    }
}
