package ru.yandex.chemodan.util.yasm.monitor;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author yashunsky
 */

public class YasmMonitor extends DelayingWorkerServiceBeanSupport {
    private static final Logger logger = LoggerFactory.getLogger(YasmMonitor.class);

    private final YasmMonitorHostsSupplier hostsSupplier;
    private final YasmAggrClient yasmClient;
    private final ListF<YasmMetricExtractor> extractors;
    private final Duration maxMeasureInterval;

    private final MapF<MetricKey, MetricAverage> metricMeters = Cf.concurrentHashMap();

    public YasmMonitor(YasmMonitorHostsSupplier hostsSupplier, YasmAggrClient yasmClient,
            ListF<YasmMetricExtractor> extractors, Duration maintenancePeriod, Duration maxMeasureInterval)
    {
        this.hostsSupplier = hostsSupplier;
        this.yasmClient = yasmClient;
        this.extractors = extractors;
        this.maxMeasureInterval = maxMeasureInterval;

        setDelay(maintenancePeriod);
    }

    private void updateHostMetrics(String host) {
        MapF<YasmMetric, Double> metrics;
        try {
            metrics = yasmClient.get(host, extractors);
        } catch (RuntimeException e) {
            logger.warn("Failed to resolve metrics for {}: {}", host, e);
            return;
        }
        metrics.forEach((key, value) -> {
            MetricKey metricKey = new MetricKey(host, key);
            metricMeters.putIfAbsent(metricKey, new MetricAverage(maxMeasureInterval));
            MetricAverage metricAverage = metricMeters.getTs(metricKey);
            metricAverage.update(value);
        });
    }

    @Override
    protected void execute() throws Exception {
        hostsSupplier.getHosts().forEach(this::updateHostMetrics);
    }

    public MapF<YasmMetric, Double> getMetrics(
            String host, ListF<YasmMetric> metrics, Duration averageInterval) throws NoRelevantValueException
    {
        return metrics.zipWith(metric -> {
            MetricKey metricKey = new MetricKey(host, metric);
            Option<MetricAverage> metricAverage = metricMeters.getO(metricKey);

            if (!metricAverage.isPresent()) {
                throw NoRelevantValueException.metricsNotFound(host, metric);
            }

            return metricAverage.get().getAverage(averageInterval,
                    ts -> NoRelevantValueException.outOfDate(host, metric, ts));
        }).toMap();
    }

    @AllArgsConstructor
    @Data
    private static class MetricKey {
        private final String host;
        private final YasmMetric metric;
    }

    public YasmHostStatus getHostStatus(
            String host, MapF<YasmMetric, YasmMetricRanges> rangesByMetric, Duration averageInterval)
    {
        MapF<YasmMetric, Double> metrics;
        try {
            metrics = getMetrics(host, rangesByMetric.keys(), averageInterval);
        } catch (NoRelevantValueException e) {
            logger.warn(e.getMessage());
            return YasmHostStatus.CRIT;
        }

        return metrics.mapEntries((metric, value) -> rangesByMetric.getTs(metric).getHostStatus(metric, value))
                .reduceRightO(YasmHostStatus::worst).getOrElse(YasmHostStatus.CRIT);
    }

    public void executeForTest() throws Exception {
        execute();
    }
}
