package ru.yandex.chemodan.util.yasm.monitor;

import org.joda.time.Duration;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Try;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.util.http.HttpException;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
public class YasmMonitorTest {
    private final String hostA = "a.host.net";
    private final String hostB = "b.host.net";
    private final String hostC = "c.host.net";
    private final Duration MAINTENANCE_INTERVAL = Duration.millis(100);
    private final Duration MAX_AVERAGE_INTERVAL = Duration.millis(10000);
    private final Duration AVERAGE_INTERVAL = Duration.millis(200);
    private YasmMonitor monitor;

    private ListF<YasmMetricExtractor> extractors;

    @Before
    public void before() {
        YasmMonitorHostsSupplier hostsSupplier = Mockito.mock(YasmMonitorHostsSupplier.class);
        YasmAggrClient yasmClient = Mockito.mock(YasmAggrClient.class);

        extractors = Cf.list(YasmMetricExtractor.pgQueryTime(), YasmMetricExtractor.io());

        Mockito.when(hostsSupplier.getHosts()).thenReturn(Cf.list(hostA, hostB));

        Mockito.when(yasmClient.get(Mockito.anyString(), Mockito.any())).then(invocation -> {
            String host = invocation.getArgument(0);
            switch (host) {
                case hostA:
                    return Cf.map(YasmMetric.PG_QUERY_TIME, 0.1, YasmMetric.IO, 10.0);
                case hostB:
                    return Cf.map(YasmMetric.PG_QUERY_TIME, 0.3, YasmMetric.IO, 14.0);
                case hostC:
                    return Cf.map(YasmMetric.PG_QUERY_TIME, 0.3, YasmMetric.IO, 20.0);
                default:
                    throw new HttpException(500);
            }
        });

        monitor = new YasmMonitor(hostsSupplier, yasmClient, extractors, MAINTENANCE_INTERVAL, MAX_AVERAGE_INTERVAL);
    }

    @Test
    public void getMetrics() throws Exception {
        Function0<MapF<YasmMetric, Double>> getMetrics = () -> monitor.getMetrics(
                hostA, extractors.map(YasmMetricExtractor::getMetric), AVERAGE_INTERVAL);

        monitor.executeForTest();
        MapF<YasmMetric, Double> metrics = getMetrics.apply();
        Assert.hasSize(2, metrics);
        Assert.equals(0.1, metrics.getTs(YasmMetric.PG_QUERY_TIME), 0.001);
        Assert.equals(10.0, metrics.getTs(YasmMetric.IO), 0.001);

        ThreadUtils.sleep(AVERAGE_INTERVAL.multipliedBy(2));

        Assert.failure(Try.tryCatchException(getMetrics), NoRelevantValueException.class);
    }

    @Test
    public void getStatus() throws Exception {
        monitor.executeForTest();

        MapF<YasmMetric, YasmMetricRanges> rangesByMetric = Cf.map(
                YasmMetric.PG_QUERY_TIME, new YasmMetricRanges(0.2, 0.4),
                YasmMetric.IO, new YasmMetricRanges(12, 15)
        );

        Assert.equals(YasmHostStatus.OK, monitor.getHostStatus(hostA, rangesByMetric, AVERAGE_INTERVAL));
        Assert.equals(YasmHostStatus.WARN, monitor.getHostStatus(hostB, rangesByMetric, AVERAGE_INTERVAL));
        Assert.equals(YasmHostStatus.CRIT, monitor.getHostStatus(hostC, rangesByMetric, AVERAGE_INTERVAL));
    }
}
