package ru.yandex.chemodan.util.yql;

import java.io.IOException;

import org.apache.http.Header;
import org.apache.http.HttpRequest;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;
import org.apache.http.util.EntityUtils;

import ru.yandex.chemodan.util.exception.PermanentFailureWithCodeException;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.parse.BenderParserException;
import ru.yandex.misc.lang.Validate;

/**
 * @author yashunsky
 */
public class YqlClient {
    private final String uri;
    private final HttpClient httpClient;
    private final Header oAuthHeader;

    BenderMapper mapper = new BenderMapper();

    private void validateId(String id) {
        Validate.isTrue(id.matches("^[a-zA-Z0-9]+$"));
    }

    public YqlClient(String uri, String token, HttpClient httpClient) {
        this.uri = uri;
        this.httpClient = httpClient;
        this.oAuthHeader = new BasicHeader("Authorization", "OAuth " + token);
    }

    private void addOauth(HttpRequest request) {
        request.addHeader(oAuthHeader);
    }

    private YqlResponse execute(HttpUriRequest request) {
        try {
            return httpClient.execute(request, (response) -> {
                String responseBody = EntityUtils.toString(response.getEntity());
                try {
                    return mapper.parseJson(YqlResponse.class, responseBody);
                } catch (BenderParserException e) {
                    YqlError error = mapper.parseJson(YqlError.class, responseBody);
                    throw new PermanentFailureWithCodeException(error.message, error.code);
                }
            });
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public YqlResponse run(String query) {
        HttpPost request = new HttpPost(uri);
        addOauth(request);
        String body = new String(mapper.serializeJson(new YqlRequest(query)));
        request.setEntity(new StringEntity(body, ContentType.APPLICATION_JSON));
        return execute(request);
    }

    public YqlResponse poll(String id) {
        validateId(id);
        HttpGet request = new HttpGet(uri + "/" + id);
        addOauth(request);
        return execute(request);
    }

    public YqlResponse getData(String id) {
        //TODO needs data support in YqlResponse
        validateId(id);
        HttpGet request = new HttpGet(uri + "/" + id + "/results?filters=DATA");
        addOauth(request);
        return execute(request);
    }

}
