package ru.yandex.chemodan.util.yt;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.operations.Operation;
import ru.yandex.inside.yt.kosher.operations.specs.CommandSpec;
import ru.yandex.inside.yt.kosher.operations.specs.MapReduceSpec;
import ru.yandex.inside.yt.kosher.tables.types.JacksonTableEntryType;
import ru.yandex.misc.log.mlf.Logger;

public abstract class BaseIncrementalLogMrYtRunner extends YtRunnerBase {

    private final ListF<String> reduceKeys;
    private final Class<?> classNearScripts;
    private final boolean uploadScriptsToYt;

    public BaseIncrementalLogMrYtRunner(Yt yt, RetryPolicy retryPolicy, YPath rootPath, String scriptName,
                                        ListF<String> reduceKeys, Class<?> classNearScripts,
                                        YtCommandParameters commandParameters,
                                        boolean uploadScriptsToYt)
    {
        super(yt, retryPolicy, rootPath, scriptName, commandParameters);
        this.reduceKeys = reduceKeys;
        this.classNearScripts = classNearScripts;
        this.uploadScriptsToYt = uploadScriptsToYt;
    }

    public Option<Operation> retrieveToday(LocalDate today) {
        return retrieve(today, 0);
    }

    public Option<Operation> retrieveFromPrevCall(LocalDate today) {
        return retrieve(today, -1);
    }

    public boolean tableReady(LocalDate by) {
        YPath path = outputPath().child(by.toString());
        return yt.existsWithRetries(path) && yt.getWithRetries(() -> yt.getRowCount(path)) > 0;
    }

    public Option<Operation> retrieve(LocalDate by, int daysBefore) {
        CommandSpec mapCommandSpec = getMapCommandSpec();

        CommandSpec reduceCommandSpec = getReduceCommandSpec();

        YPath newPath = outputPath().child(by.toString());
        if (tableReady(by)) {
            getLogger().info("Skipping {} as it is already not empty", newPath);
            return Option.empty();
        }

        ListF<YPath> inputPaths = getInputPaths(by, daysBefore);

        if (uploadScriptsToYt) {
            uploadScripts(classNearScripts);
        }

        return Option.of(yt.getWithRetries(() -> yt.operations().mapReduceAndGetOp(MapReduceSpec.builder()
                .setInputTables(inputPaths)
                .setOutputTables(Cf.list(newPath))
                .setMapperSpec(mapCommandSpec)
                .setReduceBy(reduceKeys)
                .setReducerSpec(reduceCommandSpec)
                .build())));
    }

    protected CommandSpec getMapCommandSpec() {
        return consPythonCmd(getMapParameters(), new JacksonTableEntryType(),
                new JacksonTableEntryTypeWithoutUtfEncoding()
        );
    }

    protected CommandSpec getReduceCommandSpec() {
        return consPythonCmdWithoutUtfEncoding("reduce");
    }

    protected abstract YPath outputPath();

    protected abstract ListF<YPath> getInputPaths(LocalDate by, int daysBefore);

    protected abstract Logger getLogger();

    protected ListF<String> getAdditionalMapParameters() {
        return Cf.list();
    }

    protected String getMapParameters() {
        return Cf.list("map").plus(getAdditionalMapParameters()).mkString(" ");
    }
}
