package ru.yandex.chemodan.util.yt;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.LocalDate;
import org.joda.time.Period;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */

@RunWith(MockitoJUnitRunner.class)
public class ResolveNotEmptyTablesTest {
    @Mock
    private Yt yt;
    @Mock
    private Cypress cypress;
    @Mock
    private YTreeNode yTreeNode;
    @Mock
    private YTreeNode tableNode;
    @Captor
    private ArgumentCaptor<YPath> tableDeleteCaptor;

    private YtCleaner ytCleaner;

    @Before
    public void before() {
        RetryPolicy rt = new RetryPolicy();
        rt.withMaxRetries(1);
        ytCleaner = new YtCleaner(yt, rt, Period.days(1));

        Mockito.when(yt.cypress()).thenReturn(cypress);
        Mockito.when(cypress.exists(Mockito.any(YPath.class))).thenReturn(true);
        Mockito.when(cypress.get(Mockito.any(YPath.class))).thenReturn(yTreeNode).thenReturn(tableNode);
    }

    @Test
    public void getLatestNotEmptyTableAndDeleteOld() {
        Option<YPath> latestNotEmpty = deleteOldAndGetLatestNotEmpty(
                Cf.list("2017-01-01", "2017-01-02", "2017-01-03", "2017-01-04", "2017-01-05"),
                LocalDate.parse("2017-01-04"),
                0L, 0L, 1L); // 2017-01-05 shouldn't be checked because is in future,
                             // and 2017-01-01 — because 2017-01-02 is not empty

        Assert.isTrue(latestNotEmpty.isSome(YPath.simple("//root/2017-01-02")));

        Mockito.verify(cypress, Mockito.atLeastOnce()).remove(tableDeleteCaptor.capture());

        ListF<YPath> deletedPaths = Cf.x(tableDeleteCaptor.getAllValues());
        Assert.hasSize(1, deletedPaths);
        Assert.assertContains(deletedPaths, YPath.simple("//root/2017-01-01"));
    }

    @Test
    public void keepOldEmptyTable() {
        Option<YPath> latestNotEmpty = deleteOldAndGetLatestNotEmpty(
                Cf.list("2017-01-01", "2017-01-02"),
                LocalDate.parse("2017-01-20"),
                0L, 1L);

        Assert.isTrue(latestNotEmpty.isSome(YPath.simple("//root/2017-01-01")));

        Mockito.verify(cypress, Mockito.atLeastOnce()).remove(tableDeleteCaptor.capture());

        ListF<YPath> deletedPaths = Cf.x(tableDeleteCaptor.getAllValues());
        Assert.hasSize(1, deletedPaths);
        Assert.assertContains(deletedPaths, YPath.simple("//root/2017-01-02"));
    }

    private Option<YPath> deleteOldAndGetLatestNotEmpty(ListF<String> tableNames,
            LocalDate now, Long prevTableSize, Long... prevTableSizes) {
        Mockito.when(yTreeNode.asMap()).thenReturn(tableNames.toMap(n -> n, n -> yTreeNode));
        Mockito.when(tableNode.longValue()).thenReturn(prevTableSize, prevTableSizes);
        return ytCleaner.deleteOldAndGetLatestNotEmpty(YPath.simple("//root"), now);
    }

    @After
    public void after() {
        Mockito.reset(yt, cypress, yTreeNode, tableNode);
    }

}
