package ru.yandex.chemodan.util.yt;

import java.util.Collections;
import java.util.Set;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.Duration;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.util.date.LocalDateRange;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.acl.YtAcl;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.files.YtFiles;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.operations.YtOperations;
import ru.yandex.inside.yt.kosher.tables.YtTables;
import ru.yandex.inside.yt.kosher.transactions.YtTransactions;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.time.TimeUtils;

public class YtHelper implements Yt {

    private static final Logger logger = LoggerFactory.getLogger(YtHelper.class);

    private final Yt yt;

    private final RetryPolicy retryPolicy;

    public YtHelper(Yt yt, RetryPolicy retryPolicy) {
        this.yt = yt;
        this.retryPolicy = retryPolicy;
    }

    public Function<YPath, Option<YPath>> resolveTableF(LocalDate by) {
        return path -> getLatestNotEmptyPath(path, by);
    }

    private <T> RetryManager<T> withRetries() {
        return new RetryManager<T>().withRetryPolicy(retryPolicy);
    }

    public void uploadScriptsWithRetries(Class<?> clazz, ListF<YPath> scriptPaths) {
        runWithRetries(() -> uploadScripts(clazz, scriptPaths));
    }

    private void uploadScripts(Class<?> clazz, ListF<YPath> scriptPaths) {
        scriptPaths.forEach(path -> {
            yt.cypress().create(path, CypressNodeType.FILE,
                    true, true, Cf.map("executable", YTree.booleanNode(true)));

            yt.files().write(path, new ClassPathResourceInputStreamSource(clazz, path.name()));
        });
    }

    public void remove(YPath path) {
        if (yt.cypress().exists(path)) {
            yt.cypress().remove(path);
        }
    }

    public ListF<LocalDate> getExistingTables(YPath path) {
        return Cf.wrap(getTableNames(path))
                .filterMap(TimeUtils.localDate::parseSafe)
                .toList()
                .sorted();
    }

    public Set<String> getTableNames(YPath path) {
        return getWithRetries(() -> yt.cypress().exists(path)
                ? yt.cypress().get(path).asMap().keySet()
                : Collections.emptySet());
    }

    public Tuple2<ListF<LocalDate>, Option<LocalDate>> getExistingDatesAndLatestNotEmptyDate(YPath path, LocalDate by) {
        ListF<LocalDate> existingTables = getExistingTables(path).filter(date -> !date.isAfter(by));
        return Tuple2.tuple(existingTables, getLastNotEmptyTableDate(path, existingTables));
    }

    public Option<LocalDate> getLastNotEmptyTableDate(YPath path, ListF<LocalDate> dates) {
        ListF<LocalDate> innerDates = dates.toList();
        Option<LocalDate> lastDate;
        while (innerDates.size() > 0) {
            lastDate = innerDates.lastO();
            if (lastDate.map(date -> getRowCount(path, date)).getOrElse(0L) > 0) {
                return lastDate;
            }
            innerDates = innerDates.rdrop(1);
        }

        return Option.empty();
    }

    public Option<YPath> getLatestNotEmptyPath(YPath path, LocalDate by) {
        return getExistingDatesAndLatestNotEmptyDate(path, by)._2.map(date -> path.child(date.toString()));
    }

    public long getRowCount(YPath path) {
        return getWithRetries(() -> yt.cypress().get(path.attribute("row_count")).longValue());
    }

    public long getRowCount(YPath path, LocalDate date) {
        return getRowCount(path.child(date.toString()));
    }

    @Override
    public Cypress cypress() {
        return yt.cypress();
    }

    @Override
    public YtAcl acl() {
        return yt.acl();
    }

    @Override
    public YtFiles files() {
        return yt.files();
    }

    @Override
    public YtTables tables() {
        return yt.tables();
    }

    @Override
    public YtOperations operations() {
        return yt.operations();
    }

    @Override
    public YtTransactions transactions() {
        return yt.transactions();
    }

    public ListF<YPath> getDailyLogPaths(YPath parentPath, LocalDate by, int daysBefore) {
        return Cf.range(0, daysBefore + 1)
                .map(d -> parentPath.child(by.minusDays(d).toString()))
                .filter(this::existsWithRetries);
    }

    public Boolean existsWithRetries(YPath path) {
        return getWithRetries(() -> cypress().exists(path));
    }

    public void waitForTableForMinutes(YPath table, int minutesToWait) {
        for (int i = 0; i < minutesToWait; i++) {
            if (existsWithRetries(table)) {
                logger.debug("The table {} has been found. Retry number is {}", table, i + 1);
                break;
            }
            if (i == minutesToWait - 1) { // if the retry limit has been exceeded we do not need to sleep
                throw new IllegalStateException(String.format("The table %s has not been found. Retries count is %s",
                        table, minutesToWait));
            }
            logger.debug("The table {} has not been found. Retry number is {}. Sleeping for a minute", table, i + 1);
            ThreadUtils.doSleep(Duration.standardMinutes(1));
        }
    }

    public void runWithRetries(Function0V function) {
        withRetries().run(function);
    }

    public <T> T getWithRetries(Function0<T> runnable) {
        return this.<T>withRetries()
                .get(runnable);
    }

    public ListF<DateRangeTablePath> getDateRangePaths(YPath path) {
        return Cf.wrap(getTableNames(path))
                .filterMap(LocalDateRange::parseSafe)
                .map(dateRange -> new DateRangeTablePath(this, path, dateRange));
    }

    public DailyTablePath getDailyTablePath(YPath parentPath, LocalDate date) {
        return new DailyTablePath(this, parentPath, date);
    }

    public boolean isTableEmpty(YPath path) {
        return !existsWithRetries(path) || getWithRetries(() -> getRowCount(path)) == 0;
    }
}
