package ru.yandex.chemodan.util.yt;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.operations.specs.CommandSpec;
import ru.yandex.inside.yt.kosher.operations.specs.MapReduceSpec;
import ru.yandex.inside.yt.kosher.operations.specs.MapSpec;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class YtSpecBuilder {
    private final ListF<String> commandPrefix;

    private final YPath scriptPath;

    private final ListF<YPath> inputTables = Cf.arrayList();

    private final ListF<YPath> outputTables = Cf.arrayList();

    private YTableEntryType commonEntryType = YTableEntryTypes.YSON;

    private YtSpecBuilder(ListF<String> commandPrefix, YPath scriptPath) {
        this.commandPrefix = commandPrefix;
        this.scriptPath = scriptPath;
    }

    public static YtSpecBuilder python(YPath scriptPath) {
        return new YtSpecBuilder(Cf.arrayList("python", scriptPath.name()), scriptPath);
    }

    public YtSpecBuilder inputTables(ListF<YPath> paths) {
        inputTables.addAll(paths);
        return this;
    }

    public YtSpecBuilder outputTables(ListF<YPath> paths) {
        outputTables.addAll(paths);
        return this;
    }

    public YtSpecBuilder inputTable(YPath path) {
        inputTables.add(path);
        return this;
    }

    public YtSpecBuilder outputTable(YPath path) {
        outputTables.add(path);
        return this;
    }

    public YtSpecBuilder entryType(YTableEntryType entryType) {
        this.commonEntryType = entryType;
        return this;
    }

    public MapBuilder map(String command) {
        return new MapBuilder(command);
    }

    private abstract class CommandBuilder<T extends CommandBuilder<T>> {
        private final ListF<String> command;

        private YTableEntryType inputEntryType = commonEntryType;

        private YTableEntryType outputEntryType = commonEntryType;

        private CommandBuilder(String command) {
            this.command = Cf.arrayList(command);
        }

        abstract T self();

        public T params(Object... params) {
            return params(Cf.list(params));
        }

        public T params(ListF<Object> params) {
            this.command.addAll(params.map(Object::toString));
            return self();
        }

        public T inputEntryType(YTableEntryType inputType) {
            this.inputEntryType = inputType;
            return self();
        }

        public T outputEntryType(YTableEntryType outputEntryType) {
            this.outputEntryType = outputEntryType;
            return self();
        }

        protected CommandSpec buildCommand() {
            return CommandSpec.builder()
                    .setCommand(commandPrefix.plus(command).mkString(" "))
                    .setInputType(inputEntryType)
                    .setOutputType(outputEntryType)
                    .setFiles(Cf.list(scriptPath))
                    .build();
        }
    }

    public class MapBuilder extends CommandBuilder<MapBuilder> {
        private MapBuilder(String command) {
            super(command);
        }

        @Override
        MapBuilder self() {
            return this;
        }

        public MapReduceBuilder reduce(String command) {
            return new MapReduceBuilder(command, this);
        }

        public MapSpec build() {
            return new MapSpec(inputTables, outputTables, buildCommand());
        }
    }

    public class MapReduceBuilder extends CommandBuilder<MapReduceBuilder> {
        private final MapBuilder mapBuilder;

        private final ListF<String> reduceBy = Cf.arrayList();

        private MapReduceBuilder(String command, MapBuilder mapBuilder) {
            super(command);
            this.mapBuilder = mapBuilder;
        }

        @Override
        MapReduceBuilder self() {
            return this;
        }

        public MapReduceBuilder by(ListF<String> keys) {
            reduceBy.addAll(keys);
            return this;
        }

        public MapReduceSpec build() {
            return MapReduceSpec.builder()
              .setInputTables(inputTables)
              .setOutputTables(outputTables)
              .setMapperSpec(mapBuilder.buildCommand())
              .setReduceBy(reduceBy)
              .setReducerSpec(buildCommand())
              .build();
        }
    }
}
